# id3fs - a FUSE-based filesystem for browsing audio metadata
# Copyright (C) 2010  Ian Beckwith <ianb@erislabs.net>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

package ID3FS::Path;

use strict;
use warnings;
use ID3FS::Path::Node;

our ($STATE_INVALID, $STATE_ROOT, $STATE_TAG, $STATE_BOOLEAN,
     $STATE_ALBUMS, $STATE_TRACKLIST, $STATE_FILE, $STATE_ALL)=(0..7);

# operator precedence
our %priorities=( "OR" => 0, "AND" => 1, "NOT" => 2 );

our $PATH_ALLTRACKS= "TRACKS";
our $PATH_NOARTIST = "NOARTIST";
our $PATH_NOALBUM  = "NOALBUM";

our $ENABLE_FILTER = 0;

sub new
{
    my $proto=shift;
    my $class=ref($proto) || $proto;
    my $self={};
    bless($self,$class);

    $self->{elements}=[];
    $self->{db}=shift;
    $self->{path}=shift;
    $self->{verbose}=shift;
    $self->{maxtagdepth}=shift;
    $self->{curtagdepth}=0;
    $self->{path} =~ s/\/\//\//g; # drop doubled slashes

    $self->parse();
    return $self;
}

sub isdir
{
    my($self)=@_;
    if(($self->state() == $STATE_FILE) ||
       ($self->state() == $STATE_INVALID))
    {
	return 0;
    }
    return 1;
}

sub isfile
{
    my($self)=@_;
    return($self->state() == $STATE_FILE);
}

sub isvalid
{
    my($self)=@_;
    return($self->state() != $STATE_INVALID);
}

sub dest
{
    my($self, $mountpoint)=@_;
    if($self->state() == $STATE_FILE)
    {
	return $self->filename($mountpoint);
    }
    return "ERROR"; #should never happen?
}

sub dirents
{
    my($self)=@_;
    my @dents=();
    my @fents=();
    my $state=$self->state();
    if($state==$STATE_ALL)
    {
	@dents=($self->filter($PATH_ALLTRACKS, $PATH_NOARTIST), $self->artists());
    }
    elsif($state==$STATE_TAG)
    {
	my $tail=$self->tail();
	if($self->is($TYPE_TAG, $tail) && $self->{db}->tag_has_values($tail->id()))
	{
	    @dents=$self->tags();
	}
	else
	{
	    if($self->{maxtagdepth} && ($self->{curtagdepth} < $self->{maxtagdepth}))
	    {
		@dents=qw(AND OR);
	    }
	    push(@dents, $self->filter($PATH_ALLTRACKS, $PATH_NOARTIST), $self->artists());
	}
    }
    elsif($state==$STATE_BOOLEAN)
    {
	my $parent=$self->tail();
	unless($self->is($TYPE_BOOL, $parent) && $parent->name() eq "NOT")
	{
	    @dents=("NOT");
	}
	push(@dents,$self->tags());
    }
    elsif($state==$STATE_ROOT)
    {
	@dents=(qw(ALL NOT), $self->tags());
    }
    elsif($state==$STATE_ALBUMS)
    {
	@dents=$self->filter($PATH_ALLTRACKS, $PATH_NOALBUM, $self->albums());
    }
    elsif($state==$STATE_TRACKLIST)
    {
	@fents=$self->tracks();
    }
    else
    {
	print "id3fsd: INTERNAL ERROR: DIRENTS: UNHANDLED STATE: $state\n";
    }
    return(\@dents, \@fents);
}

# State Machine Of Doom
sub parse
{
    my($self)=@_;
    $self->state($STATE_ROOT);
    return if($self->{path} eq "/");
    @{$self->{components}}=split(/\//, $self->{path});
    shift @{$self->{components}}; # drop empty field before leading /
    my @parts=@{$self->{components}};
    my($tag, $tagval);
    $self->{elements}=[];
    $self->{in_all}=0;
    my $root_not=0;
    my $tags_seen=0;
    while(defined(my $name=shift @parts))
    {
	my $state=$self->state();
	if($state==$STATE_INVALID)
	{
	    return;
	}
	elsif($state==$STATE_ROOT)
	{
	    if($name eq "ALL")
	    {
		$self->{in_all}=1;
		$self->state($STATE_ALL);
	    }
	    elsif($name eq "NOT")
	    {
		$root_not=1;
	        push(@{$self->{elements}}, ID3FS::Path::Node->new($self->{db}, $TYPE_BOOL, $name));
		$self->state($STATE_BOOLEAN);
	    }
	    else
	    {
		$tag=ID3FS::Path::Node->new($self->{db}, $TYPE_TAG, $name);
		if($tag)
		{
		    push(@{$self->{elements}}, $tag);
		    $tags_seen++;
		    $self->state($STATE_TAG);
		}
		else
		{
		    $self->state($STATE_INVALID);
		}
	    }
	}
	elsif($state==$STATE_TAG)
	{
	    my $tag=$self->tail();
	    if($self->is($TYPE_TAG, $tag) && $self->{db}->tag_has_values($tag->id()))
	    {
		my $tagval=ID3FS::Path::Node->new($self->{db}, $TYPE_TAG, $name, $tag->id());
		if(defined($tagval))
		{
		    # stay in tag state
		    push(@{$self->{elements}}, $tagval);
		}
		else
		{
		    $self->state($STATE_INVALID);
		}
	    }
	    elsif($name eq $PATH_ALLTRACKS)
	    {
		$self->state($STATE_TRACKLIST);
	    }
	    elsif($name eq $PATH_NOARTIST)
	    {
		$self->state($STATE_TRACKLIST);
	    }
	    elsif($name eq "AND")
	    {
		$self->state($STATE_BOOLEAN);
	        push(@{$self->{elements}}, ID3FS::Path::Node->new($self->{db}, $TYPE_BOOL, $name));
	    }
	    elsif($name eq "OR")
	    {
		$self->state($STATE_BOOLEAN);
	        push(@{$self->{elements}}, ID3FS::Path::Node->new($self->{db}, $TYPE_BOOL, $name));
	    }
	    else
	    {
		my $artist=ID3FS::Path::Node->new($self->{db}, $TYPE_ARTIST, $name);
		if($artist)
		{
		    push(@{$self->{elements}}, $artist);
		    $self->state($STATE_ALBUMS);
		}
		else
		{
		    $self->state($STATE_INVALID);
		}
	    }
	}
	elsif($state==$STATE_BOOLEAN)
	{
	    my $parent=$self->tail();
	    my $allownot=1;
	    if($self->is($TYPE_BOOL, $parent) &&
	       $parent->name() eq "NOT")
	    {
		$allownot=0;
	    }
	    if($allownot && $name eq "NOT")
	    {
		$self->state($STATE_BOOLEAN);
	        push(@{$self->{elements}}, ID3FS::Path::Node->new($self->{db}, $TYPE_BOOL, $name));
	    }
	    else
	    {
		my $tag=ID3FS::Path::Node->new($self->{db}, $TYPE_TAG, $name);
		if($tag)
		{
		    push(@{$self->{elements}}, $tag);
		    $tags_seen++;
		    $self->state($STATE_TAG);
		}
		else
		{
		    $self->state($STATE_INVALID);
		}
	    }
	}
	elsif($state==$STATE_ALBUMS)
	{
	    if($name eq $PATH_ALLTRACKS)
	    {
		$self->state($STATE_TRACKLIST);
	    }
	    elsif($name eq $PATH_NOALBUM)
	    {
		$self->state($STATE_TRACKLIST);
	    }
	    else
	    {
		my $album=ID3FS::Path::Node->new($self->{db}, $TYPE_ALBUM, $name);
		if($album)
		{
		    push(@{$self->{elements}}, $album);
		    $self->state($STATE_TRACKLIST);
		}
		else
		{
		    $self->state($STATE_INVALID);
		}
	    }
	}
	elsif($state==$STATE_TRACKLIST)
	{
	    my $track=ID3FS::Path::Node->new($self->{db}, $TYPE_FILE, $name);
	    if($track)
	    {
		push(@{$self->{elements}}, $track);
		$self->state($STATE_FILE);
	    }
	    else
	    {
		$self->state($STATE_INVALID);
	    }
	}
	elsif($state==$STATE_FILE)
	{
	    # Can't have anything after a filename
	    $self->state($STATE_INVALID);
	}
	elsif($state==$STATE_ALL)
	{
	    if($name eq $PATH_ALLTRACKS)
	    {
		$self->state($STATE_TRACKLIST);
	    }
	    elsif($name eq $PATH_NOARTIST)
	    {
		$self->state($STATE_TRACKLIST);
	    }
	    else
	    {
		my $artist=ID3FS::Path::Node->new($self->{db}, $TYPE_ARTIST, $name);
		if($artist)
		{
		    push(@{$self->{elements}}, $artist);
		    $self->state($STATE_ALBUMS);
		}
		else
		{
		    $self->state($STATE_INVALID);
		}
	    }
	}
	else
	{
	    print "SM: ERROR: UNKNOWN STATE: $self->{state}\n";
	    $self->state($STATE_INVALID);
	}
    }

    my @elements=@{$self->{elements}};
    # remove trailing boolean
    while(@elements && $self->is($TYPE_BOOL, $elements[$#elements]))
    {
	pop @elements;
    }

    # calculate joins and assign table numbers to nodes
    $self->{joins}=[$self->number_joins(@elements)];

    # always need at least one join
    $self->{joins}=[qw(INNER)] unless(@{$self->{joins}});

    # sort elements by precedence
    @elements=$self->sort_elements(@elements);

    # convert to binary tree
    $self->{tagtree}=$self->elements_to_tree(\@elements);
}

sub number_joins
{
    my($self, @elements)=@_;
    my @joins=("INNER");
    my $table=1;
    my $nextjoin=undef;
    my $lastop=undef;
    return (@joins) unless(@elements);
    while(my $node=shift @elements)
    {
	if($node->type() == $TYPE_BOOL)
	{
	    my $op=$node->name();
	    if($op eq "AND")
	    {
		$nextjoin="INNER";
	    }
	    elsif($op eq "NOT")
	    {
		$nextjoin="LEFT";
	    }
	    elsif($op eq "OR")
	    {
		# NOT/foo/OR needs an extra join
		$nextjoin="INNER" if($lastop && $lastop eq "NOT");
	    }
	    $lastop=$op;
	}
	else
	{
	    $node->hasvals(0);
	    if(@elements)
	    {
		# if tag has a value, eat the tag, shifting to the value
		if($elements[0]->type() == $TYPE_TAG)
		{
		    $node->hasvals(1);
		    $node=shift(@elements);
		}
	    }
	    elsif($self->{db}->tag_has_values($node->id()))
	    {
		# if the expression ends in a tag that has a value
		# (ie we have the tag and want the value)
		# use an INNER join even if we were in a NOT
		$nextjoin="INNER" if($nextjoin);
	    }
	    if($nextjoin)
	    {
		$table++;
		push(@joins, $nextjoin);
		$nextjoin=undef;
	    }
	    $node->table($table);
	}
    }
    return @joins;
}

sub state
{
    my($self, $newstate)=@_;
    if(defined($newstate))
    {
	$self->{state}=$newstate;
	$self->{curtagdepth}++ if($newstate == $STATE_TAG);
    }
    return $self->{state};
}

# link up precedence-sorted list into a binary tree
sub elements_to_tree
{
    my($self, $elements)=@_;
    return undef unless(@$elements);
    my ($left, $right, $op)=(undef, undef, undef);
    my $node=pop @$elements;
    if($self->is($TYPE_BOOL, $node))
    {
	$right=$self->elements_to_tree($elements);
	if($node->name() ne "NOT")
	{
	    $left=$self->elements_to_tree($elements);
	}
	$node->left($left);
	$node->right($right);
    }
    return $node;
}

# Dijkstra's shunting-yard algorithm
sub sort_elements
{
    my ($self, @input)=@_;
    my @opstack=();
    my @output=();
    while(my $node = shift @input)
    {
	if($self->is($TYPE_TAG, $node))
	{
	    # Handle tag values by dropping parent
	    if(@input && $self->is($TYPE_TAG, $input[0]))
	    {
		$node=shift @input;
	    }
	    push(@output, $node);
	}
	elsif($self->is($TYPE_BOOL, $node))
	{
	    while(@opstack &&
		  ($priorities{$node->name()} <= $priorities{$opstack[$#opstack]->name()}))
	    {
		push(@output, pop(@opstack));
	    }
	    push(@opstack, $node);
	}
    }
    while(@opstack)
    {
	push(@output, pop(@opstack));
    }
    return @output;
}

sub used_tags
{
    my($self)=@_;
    return() unless(defined($self->{tagtree}));
    return($self->{tagtree}->used_tags());
}

sub expecting_values
{
    my($self)=@_;
    my $tail=$self->tail();
    if($self->is($TYPE_TAG, $tail))
    {
	return($self->{db}->tag_has_values($tail->id()));
    }
}

sub trailing_tag_id
{
    my($self)=@_;
    my $tail=$self->tail();
    if($self->is($TYPE_TAG, $tail))
    {
	return($tail->id());
    }
    return undef;
}

sub trailing_tag_parent
{
    my($self)=@_;
    my $tail=$self->tail();
    if($self->is($TYPE_TAG, $tail))
    {
	return($tail->{parents_id});
    }
    return undef;
}

sub tail
{
    my($self)=@_;
    return($self->{elements}->[$#{$self->{elements}}]);
}

sub is
{
    my($self, $type, $node)=@_;
    return 0 unless($node);
    return 0 unless($node->type());
    return 1 if($type == $node->type());
    return 0;
}

# the one before last
sub tail_parent
{
    my($self)=@_;
    return($self->{elements}->[($#{$self->{elements}}) - 1]);
}

######################################################################

sub tags
{
    my($self)=@_;
    if(!$self->{tagtree}) # / or /NOT
    {
	my $sql="SELECT DISTINCT name FROM tags WHERE parents_id='';";
	return($self->{db}->cmd_firstcol($sql));
    }
    my $sql="SELECT tags.name FROM ";
    if($self->want_all_tags())
    {
	$sql .= "files_x_tags\n";
    }
    else
    {
	$sql .= ("(\n" .
		 $self->tags_subselect() .
		 ") AS subselect\n" .
		 "INNER JOIN files_x_tags ON subselect.files_id=files_x_tags.files_id\n");
    }
    $sql .= "INNER JOIN tags ON files_x_tags.tags_id=tags.id\n";
    my @andclauses=();
    my $id=$self->trailing_tag_id();

    my $parentclause= "tags.parents_id='";
    $parentclause .= $id if($self->expecting_values());
    $parentclause .= "'";
    push(@andclauses, $parentclause);

    my @used=$self->used_tags();
    if(@used)
    {
	push(@andclauses, "tags.id NOT IN (" . join(', ', @used) . ")");
    }
    if(@andclauses)
    {
	$sql .= "WHERE " . join(' AND ', @andclauses) . "\n";
    }

    $sql .= "GROUP BY tags.name;";
    print "SQL(TAGS): $sql\n" if($self->{verbose});
    my @tagnames=$self->{db}->cmd_firstcol($sql);
    print("SUBNAMES: ", join(', ', @tagnames), "\n") if($self->{verbose});
    return(@tagnames);
}

sub artists
{
    my($self)=@_;
    if(!@{$self->{elements}}) # /ALL
    {
	my $sql="SELECT DISTINCT name FROM artists WHERE name!='';";
	return($self->{db}->cmd_firstcol($sql));
    }
    my $sql=$self->sql_start("artists.name");
    $sql .= ("INNER JOIN artists ON files.artists_id=artists.id\n" .
	     "WHERE artists.name != ''\n" .
	     "GROUP BY artists.name;");
    print "SQL(ARTISTS): $sql\n" if($self->{verbose});
    my @tagnames=$self->{db}->cmd_firstcol($sql);
    print("ARTISTS: ", join(', ', @tagnames), "\n") if($self->{verbose});
    return(@tagnames);
}

sub albums
{
    my($self)=@_;
    my $tail=$self->tail();
    if($self->is($TYPE_ARTIST, $tail))
    {
	return $self->artist_albums($tail->id());
    }
    my $sql=$self->sql_start("albums.name");
    $sql .= ("INNER JOIN albums ON files.albums_id=albums.id\n" .
	     "WHERE albums.name != ''\n" .
	     "GROUP BY albums.name;");
    print "SQL(ALBUMS): \n$sql\n" if($self->{verbose});
    my @names=$self->{db}->cmd_firstcol($sql);
    print("ALBUMS: ", join(', ', @names), "\n") if($self->{verbose});
    return(@names);
}

sub artist_albums
{
    my($self, $artist_id)=@_;
    my $sql=$self->sql_start("albums.name");
    $sql .= ("INNER JOIN albums ON albums.id=files.albums_id\n" .
	     "INNER JOIN artists ON artists.id=files.artists_id\n" .
	     "WHERE artists.id=? and albums.name <> ''\n" .
	     "GROUP BY albums.name\n");
    print "ARTIST_ALBUMS SQL: $sql\n" if($self->{verbose});
    my @albums=$self->{db}->cmd_firstcol($sql, $artist_id);
    print("ALBUMS: ", join(', ', @albums), "\n") if($self->{verbose});
    return(@albums);
}

sub artist_tracks
{
    my($self, $artist_id)=@_;
    my $sql=$self->sql_start("files.name");
    $sql .= ("INNER JOIN artists ON artists.id=files.artists_id\n" .
	     "INNER JOIN albums  ON albums.id=files.albums_id\n" .
	     "WHERE artists.id=? AND albums.name=''\n" .
	     "GROUP BY files.name\n");
    print "ARTIST_TRACKS SQL: $sql\n" if($self->{verbose});
    my @names=$self->{db}->cmd_firstcol($sql, $artist_id);
    print("ARTISTTRACKS: ", join(', ', @names), "\n") if($self->{verbose});
    return(@names);
}

sub album_tracks
{
    my($self, $artist_id, $album_id)=@_;
    my $sql=("SELECT files.name FROM files\n" .
	     "INNER JOIN albums  ON albums.id=files.albums_id\n" .
	     "INNER JOIN artists ON artists.id=files.artists_id\n" .
	     "WHERE artists.id=? AND albums.id=?\n" .
	     "GROUP BY files.name\n");
    print "ALBUM_TRACKS SQL($artist_id, $album_id): $sql\n" if($self->{verbose});
    my @names=$self->{db}->cmd_firstcol($sql, $artist_id, $album_id);
    print("TRACKS: ", join(', ', @names), "\n") if($self->{verbose});
    return(@names);
}

sub tracks
{
    my($self)=@_;
    my $tail=$self->tail();
    if($self->is($TYPE_ARTIST, $tail))
    {
	return $self->artist_tracks($tail->id());
    }
    elsif($self->is($TYPE_ALBUM, $tail))
    {
	my $artist_id=0;
	my $artist=$self->tail_parent();
	if($self->is($TYPE_ARTIST, $artist))
	{
	    # should always happen
	    $artist_id=$artist->id();
	}
	return $self->album_tracks($artist_id, $tail->id());
    }
    my $sql=$self->sql_start("files.name");
    $sql .= "INNER JOIN artists ON files.artists_id=artists.id\n";
    if($self->{components}->[$#{$self->{components}}] eq $PATH_NOARTIST)
    {
	$sql .= "WHERE artists.name =''\n";
    }
    $sql .= "GROUP BY files.name;";
    print "TRACKS SQL($self->{path}): $sql\n" if($self->{verbose});
    my @names=$self->{db}->cmd_firstcol($sql);
    print("TRACKS: ", join(', ', @names), "\n") if($self->{verbose});
    return(@names);
}

sub filename
{
    my($self, $mountpoint)=@_;
    my $tail=$self->tail();
    if($self->is($TYPE_FILE, $tail))
    {
	my $id=$tail->id();
	my $sql=("SELECT paths.name, files.name FROM files\n" .
		 "INNER JOIN paths ON files.paths_id=paths.id\n" .
		 "WHERE files.id=?\n" .
		 "GROUP BY paths.name, files.name");
	print "FILENAME SQL: $sql\n" if($self->{verbose});
	my ($path, $name)=$self->{db}->cmd_onerow($sql, $id);
	return($self->{db}->relativise($path, $name, $mountpoint, $self->{path}));
    }
    # should never happen
    return "ERROR";
}

sub tags_subselect
{
    my($self)=@_;
    my $tree=$self->{tagtree};
    my ($sqlclause, @joins)=$tree->to_sql() if($tree);
    my $sql="\tSELECT fxt1.files_id FROM tags t1";
    my @crosses=();
    my @inners=();
    @joins=@{$self->{joins}};
    for(my $i=0; $i <= $#joins; $i++)
    {
	my $cnt=$i+1;
	my $join=$joins[$i];
	my $inner=("\t$join JOIN files_x_tags fxt$cnt ON " .
		   "t${cnt}.id=fxt${cnt}.tags_id");
	if($i > 0)
	{
	    push(@crosses, "CROSS JOIN tags t$cnt");
	    $inner .= " AND fxt1.files_id=fxt${cnt}.files_id";
	}
	push(@inners, $inner);
    }
    $sql .= ("\n\t" . join(" ", @crosses)) if(@crosses);
    $sql .= ("\n" . join("\n", @inners)) if(@inners);
    $sql .= "\n\tWHERE $sqlclause" if($sqlclause);
    $sql .= "\n\tGROUP BY fxt1.files_id\n";
    return $sql;
}

sub sql_start
{
    my($self, $tables)=@_;
    my $sql="SELECT $tables FROM ";
    if($self->{in_all})
    {
	$sql .= "files\n";
    }
    else
    {
	$sql .= ("(\n" .
		 $self->tags_subselect() .
		 ") AS subselect\n" .
		 "INNER JOIN files ON subselect.files_id=files.id\n");
    }
    return $sql;
}

# we just filter $ALLTRACKS, $NOARTIST and $NOALBUM
# filtering tags properly requires up to four levels of recursion
# (tag/tagval/AND/NOT) and is too slow
sub filter
{
    my($self, @dirs)=@_;
    return(@dirs) unless($ENABLE_FILTER);
    my $base=$self->{path};
    my @outdirs=();
    for my $dir (@dirs)
    {
	push(@outdirs, $dir) unless($self->empty("$base/$dir"));
    }
    return(@outdirs);
}

sub empty
{
    my($self, $dir)=@_;
    my $path=ID3FS::Path->new($self->{db}, $dir, $self->{verbose},
			      ($self->{maxtagdepth} - $self->{curtagdepth}));
    return 1 unless($path->isvalid());
    my($subdirs,$subfiles)=$path->dirents();
    return 0 if(@$subfiles || @$subdirs);
    return 1;
}

# if path is .../OR/ or .../OR/NOT or .../AND/NOT
sub want_all_tags
{
    my($self)=@_;
    my $tail=$self->tail();
    return 0 unless($tail);
    my $parent=$self->tail_parent();
    my $parent_valid = ($parent && $parent->type() == $TYPE_BOOL);
    if($tail->type() == $TYPE_BOOL)
    {
	return 1 if($tail->name() eq "OR");
	return 0 unless($tail->name() eq "NOT");
	return 0 unless($parent_valid);
	return 1 if($parent->name() eq "OR");
	return 1 if($parent->name() eq "AND");
    }
    elsif($tail->type() == $TYPE_TAG)
    {
	return 0 unless($parent_valid);
	return 1 if($parent->name() eq "NOT");
    }
    return 0;
}

1;
