# id3fs - a FUSE-based filesystem for browsing audio metadata
# Copyright (C) 2010  Ian Beckwith <ianb@erislabs.net>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

package ID3FS::Path::Node;

use strict;
use warnings;

require Exporter;
use vars qw(@ISA @EXPORT $TYPE_BOOL $TYPE_TAG $TYPE_ARTIST $TYPE_ALBUM $TYPE_FILE);
@ISA=qw(Exporter);
@EXPORT=qw($TYPE_BOOL $TYPE_TAG $TYPE_ARTIST $TYPE_ALBUM $TYPE_FILE);
($TYPE_BOOL, $TYPE_TAG, $TYPE_ARTIST, $TYPE_ALBUM, $TYPE_FILE)=(1..5);

sub new
{
    my $proto=shift;
    my $class=ref($proto) || $proto;
    my $self={};
    bless($self,$class);

    my $db=shift;
    $self->{type}=shift;
    $self->{name}=shift;
    $self->{parents_id}=shift;
    $self->{table}=1;
    $self->{hasvals}=undef;
    if($self->{type} != $TYPE_BOOL)
    {
	my $table='';
	if   ($self->{type} == $TYPE_TAG)    { $table="tags";    }
	elsif($self->{type} == $TYPE_ARTIST) { $table="artists"; }
	elsif($self->{type} == $TYPE_ALBUM)  { $table="albums";  }
	elsif($self->{type} == $TYPE_FILE)   { $table="files";   }
	$self->{id}=$db->lookup_id($table, $self->{name}, $self->{parents_id});
	return undef unless(defined($self->{id}));
    }
    return $self;
}

sub set
{
    my($self, $name, $val)=@_;
    if(defined($val))
    {
	$self->{$name}=$val;
    }
    return $self->{$name};
}

sub left       { return shift->set("left",       shift); }
sub right      { return shift->set("right",      shift); }
sub name       { return shift->set("name",       shift); }
sub type       { return shift->set("type",       shift); }
sub id         { return shift->set("id",         shift); }
sub parents_id { return shift->set("parents_id", shift); }
sub table      { return shift->set("table",      shift); }
sub hasvals    { return shift->set("hasvals",    shift); }

sub to_sql
{
    my($self, $not)=@_;
    $not=0 unless(defined($not));
    my $str='';

    if($self->type() != $TYPE_BOOL)
    {
	$str .= "t" . $self->table() . ".id='" . $self->{id} . "'";
	if($not && !$self->hasvals())
	{
	    $str = "(" . $str . " AND fxt" . $self->table() . ".files_id IS NULL)";
	}
	return ($str);
    }

    my $left=$self->left();
    my $right=$self->right();
    return ("") unless($left || $right);

    my $leftstr = $left->to_sql($not) if($left);
    my $op=$self->name();
    $not=1 if(defined($op) && ($op eq "NOT"));
    my $rightstr = $right->to_sql($not) if($right);
    $str = $leftstr;
    $str .= " $op " if($op && !$not);
    $str .= $rightstr;
    $str=("(" . $str . ")") if($op && $left && $right);
    return($str);
}

sub used_tags
{
    my($self)=@_;
    if($self->type() == $TYPE_BOOL)
    {
	my @used=();
	push(@used, $self->left()->used_tags())  if($self->left());
	push(@used, $self->right()->used_tags()) if($self->right());
	return(grep { defined; } @used);
    }
    return $self->id();
}

1;
