/*
 * Copyright (c) 1989, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char sccsid[] = "@(#)utility.c	8.4 (Berkeley) 5/30/95";
#endif /* not lint */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef TLS
#define SEND  tls_send
#define WRITE tls_write
#define READ  tls_read
#else
#define SEND  send
#define WRITE write
#define READ  read
#endif

#define PRINTOPTIONS
#include "telnetd.h"

#ifdef HAVE_SYS_UTSNAME_H
#include <sys/utsname.h>
#endif

/*
 * utility functions performing io related tasks
 */

/*
 * ttloop
 *
 *	A small subroutine to flush the network output buffer, get some data
 * from the network, and pass it through the telnet state machine.  We
 * also flush the pty input buffer (by dropping its data) if it becomes
 * too full.
 */

    void
ttloop()
{
    void netflush();

    DIAG(TD_REPORT, {diag_printf("td: ttloop\r\n");});
    if (nfrontp-nbackp) {
	netflush();
    }
    ncc = READ(net, netibuf, sizeof netibuf);
    if (ncc < 0) {
	syslog(LOG_INFO, "ttloop:  read: %m");
	clean_exit(1);
    } else if (ncc == 0) {
	syslog(LOG_INFO, "ttloop:  peer died: %m");
	clean_exit(1);
    }
    DIAG(TD_REPORT, {diag_printf("td: ttloop read %d chars\r\n", ncc);});
    netip = netibuf;
    telrcv();			/* state machine */
    if (ncc > 0) {
	pfrontp = pbackp = ptyobuf;
	telrcv();
    }
}  /* end of ttloop */

/*
 * Check a descriptor to see if out of band data exists on it.
 */
    int
stilloob(s)
    int	s;		/* socket number */
{
    static struct timeval timeout = { 0 };
    fd_set	excepts;
    int value;

    do {
	FD_ZERO(&excepts);
	FD_SET(s, &excepts);
	value = select(s+1, (fd_set *)0, (fd_set *)0, &excepts, &timeout);
    } while ((value == -1) && (errno == EINTR));

    if (value < 0) {
	fatalperror(pty, "select");
    }
    if (FD_ISSET(s, &excepts)) {
	return 1;
    } else {
	return 0;
    }
}

	void
ptyflush()
{
	int n;

	if ((n = pfrontp - pbackp) > 0) {
		DIAG((TD_REPORT | TD_PTYDATA),
			{ diag_printf("td: ptyflush %d chars\r\n", n); });
		DIAG(TD_PTYDATA, printdata("pd", pbackp, n));
		n = write(pty, pbackp, n);
	}
	if (n < 0) {
		if (errno == EWOULDBLOCK || errno == EINTR)
			return;
		cleanup(0);
	}
	else
		pbackp += n;
	if (pbackp == pfrontp)
		pbackp = pfrontp = ptyobuf;
}

/*
 * nextitem()
 *
 *	Return the address of the next "item" in the TELNET data
 * stream.  This will be the address of the next character if
 * the current address is a user data character, or it will
 * be the address of the character following the TELNET command
 * if the current address is a TELNET IAC ("I Am a Command")
 * character.
 */
    char *
nextitem(current)
    char	*current;
{
    if ((*current&0xff) != IAC) {
	return current+1;
    }
    switch (*(current+1)&0xff) {
    case DO:
    case DONT:
    case WILL:
    case WONT:
	return current+3;
    case SB:		/* loop forever looking for the SE */
	{
	    register char *look = current+2;

	    for (;;) {
		if ((*look++&0xff) == IAC) {
		    if ((*look++&0xff) == SE) {
			return look;
		    }
		}
	    }
	}
    default:
	return current+2;
    }
}  /* end of nextitem */


/*
 * netclear()
 *
 *	We are about to do a TELNET SYNCH operation.  Clear
 * the path to the network.
 *
 *	Things are a bit tricky since we may have sent the first
 * byte or so of a previous TELNET command into the network.
 * So, we have to scan the network buffer from the beginning
 * until we are up to where we want to be.
 *
 *	A side effect of what we do, just to keep things
 * simple, is to clear the urgent data pointer.  The principal
 * caller should be setting the urgent data pointer AFTER calling
 * us in any case.
 */
    void
netclear()
{
/* this function is not designed for the high data volume of FORWARD_X */
#ifndef FWD_X
    register char *thisitem, *next;
    char *good;
#define	wewant(p)	((nfrontp > p) && ((*p&0xff) == IAC) && \
				((*(p+1)&0xff) != EC) && ((*(p+1)&0xff) != EL))

/*XXX*/ DIAG(TD_NETDATA, { diag_printf("XXX Doing netclear() with %d bytes waiting to go!\r\n", nfrontp - nbackp); });
#ifdef	ENCRYPTION
    thisitem = nclearto > netobuf ? nclearto : netobuf;
#else	/* ENCRYPTION */
    thisitem = netobuf;
#endif	/* ENCRYPTION */

    while ((next = nextitem(thisitem)) <= nbackp) {
	thisitem = next;
    }

    /* Now, thisitem is first before/at boundary. */

#ifdef	ENCRYPTION
    good = nclearto > netobuf ? nclearto : netobuf;
#else	/* ENCRYPTION */
    good = netobuf;	/* where the good bytes go */
#endif	/* ENCRYPTION */

    while (nfrontp > thisitem) {
	if (wewant(thisitem)) {
	    int length;

	    next = thisitem;
	    do {
		next = nextitem(next);
	    } while (wewant(next) && (nfrontp > next));
	    length = next-thisitem;
	    memmove(good, thisitem, length);
	    good += length;
	    thisitem = next;
	} else {
	    thisitem = nextitem(thisitem);
	}
    }

    nbackp = netobuf;
    nfrontp = good;		/* next byte to be sent */
    neturg = 0;
#endif /* FWD_X */
}  /* end of netclear */

/*
 *  netflush
 *	Send as much data as possible to the network,
 *  handling requests for urgent data.
 *	Sorry for all the DIAG() output mess...   /Luna
 */
    void
netflush()
{
    int n;
#ifdef DIAGNOSTICS
    int nd, dpc;
#endif
    extern int not42;

    if ((n = nfrontp - nbackp) > 0) {
	DIAG(TD_REPORT | TD_NETDATA, {
	    if (neturg && not42) { nd = neturg - nbackp; if (nd > 1) nd--; }
	    else nd = nfrontp - nbackp;
	    dpc = diag_printf("td: netflush %d chars\r\n", nd); n += dpc; nd += dpc; });
#ifdef	ENCRYPTION
	/* since the data actually might get encrypted in the next block (but
	 * why not use TLS instead? ;-) ), we dump the data to the log here, and
	 * later put another notice in the log if we didn't really send it all.
	 * we also only dump the data if we use an alternative log file; dumping
	 * to the net is probably too messy... */
	DIAG(TD_NETDATA, { if (diagnostic & TD_TO_FILE) printdata("nd", nbackp, nd); });
	if (encrypt_output) {
		char *s = nclearto ? nclearto : nbackp;
		if (nfrontp - s > 0) {
			(*encrypt_output)((unsigned char *)s, nfrontp-s);
			nclearto = nfrontp;
		}
	}
#endif	/* ENCRYPTION */
retry:
	/*
	 * if no urgent data, or if the other side appears to be an
	 * old 4.2 client (and thus unable to survive TCP urgent data),
	 * write the entire buffer in non-OOB mode.
	 */
	if ((neturg == 0) || (not42 == 0)) {
	    n = WRITE(net, nbackp, n);	/* normal write */
	} else {
	    n = neturg - nbackp;
	    /*
	     * In 4.2 (and 4.3) systems, there is some question about
	     * what byte in a sendOOB operation is the "OOB" data.
	     * To make ourselves compatible, we only send ONE byte
	     * out of band, the one WE THINK should be OOB (though
	     * we really have more the TCP philosophy of urgent data
	     * rather than the Unix philosophy of OOB data).
	     */
	    if (n > 1) {
		/* here, we must restore "nd" (how much we want to send) */
		DIAG((TD_REPORT | TD_NETDATA), { nd = neturg - nbackp - 1; });
		n = SEND(net, nbackp, n - 1, 0);   /* send URGENT all by itself */
	    } else {
		/* here, we must restore "nd" (how much we want to send) */
		DIAG((TD_REPORT | TD_NETDATA), { nd = neturg - nbackp; });
		n = SEND(net, nbackp, n, MSG_OOB);	/* URGENT data */
	    }
	}
    if (n < 0) {
	if (errno == EWOULDBLOCK || errno == EINTR) {
#if 0
           fd_set wfds;
           struct timeval tv;
           int r;
           /* watch stdout (fd 1) to see when it's ready to swallow more */
           FD_ZERO(&wfds);
           FD_SET(1, &wfds);
           /* wait up to five seconds. */
           tv.tv_sec = 5; tv.tv_usec = 0;
/*           r = select(2, &wfds, NULL, NULL, &tv);*/
           r = select(2, NULL, &wfds, NULL, NULL);
           if (r)
	       goto retry;
#endif
	    /* XXX should we really accept that netflush() might return without
	     * actually doing any flushing? */
	    DIAG((TD_REPORT | TD_NETDATA),
	        { diag_printf("td: netflush failed, will retry later\r\n"); });
	    return;
	}
	DIAG((TD_REPORT | TD_NETDATA),
	    { diag_printf("td: netflush failed with errno=%d, telnetd terminating\r\n", errno); });
	cleanup(0);
    }
#ifdef ENCRYPTION
	DIAG(TD_NETDATA, { if (n >= 0 && n < nd)
	    diag_printf("nd: netsent ONLY %d chars, %d REMAIN in netobuf, but above shows ALL %d!\r\n",
			n, nd - n, nd); });
#else
	DIAG(TD_NETDATA, { if (diagnostic & TD_TO_FILE) printdata("nd", nbackp, n); });
	DIAG((TD_REPORT | TD_NETDATA), { if (n >= 0 && n < nd)
	    diag_printf("td: netsent ONLY %d chars, %d REMAIN in netobuf!\r\n", n, nd - n); });
#endif /* !ENCRYPTION */
    }
    nbackp += n;
#ifdef	ENCRYPTION
    if (nbackp > nclearto)
	nclearto = 0;
#endif	/* ENCRYPTION */
    if (nbackp >= neturg) {
	neturg = 0;
    }
    if (nbackp == nfrontp) {
	nbackp = nfrontp = netobuf;
#ifdef	ENCRYPTION
	nclearto = 0;
#endif	/* ENCRYPTION */
    }
    return;
}  /* end of netflush */


/*
 * writenet
 *
 * Just a handy little function to write a bit of raw data to the net.
 * It will force a transmit of the buffer if necessary
 *
 * arguments
 *    ptr - A pointer to a character string to write
 *    len - How many bytes to write
 */
	void
writenet(ptr, len)
	register unsigned char *ptr;
	register int len;
{
	/* flush buffer if no room for new data) */
	if ((&netobuf[netobuf_size] - nfrontp) < len) {
		/* if this fails, don't worry, buffer is a little big */
		netflush();
	}

	memmove(nfrontp, ptr, len);
	nfrontp += len;

}  /* end of writenet */


/*
 * miscellaneous functions doing a variety of little jobs follow ...
 */


	void
fatal(f, msg)
	int f;
	char *msg;
{
	char buf[BUFSIZ];

	snprintf (buf, sizeof buf, "telnetd: %s.\r\n", msg);
#ifdef	ENCRYPTION
	if (encrypt_output) {
		/*
		 * Better turn off encryption first....
		 * Hope it flushes...
		 */
		encrypt_send_end();
		netflush();
	}
#endif	/* ENCRYPTION */
	(void) write(f, buf, (int)strlen(buf));
	sleep(1);	/*XXX*/
	clean_exit(1);
}

	void
fatalperror(f, msg)
	int f;
	char *msg;
{
	char buf[BUFSIZ];

	snprintf (buf, sizeof buf, "%s: %s", msg, strerror(errno));
	fatal(f, buf);
}

char editedhost[32];

	void
edithost(pat, host)
	register char *pat;
	register char *host;
{
	register char *res = editedhost;
#ifndef strncpy
	char *strncpy();
#endif

	if (!pat)
		pat = "";
	while (*pat) {
		switch (*pat) {

		case '#':
			if (*host)
				host++;
			break;

		case '@':
			if (*host)
				*res++ = *host++;
			break;

		default:
			*res++ = *pat;
			break;
		}
		if (res == &editedhost[sizeof editedhost - 1]) {
			*res = '\0';
			return;
		}
		pat++;
	}
	if (*host)
		(void) strncpy(res, host,
				sizeof editedhost - (res - editedhost) -1);
	else
		*res = '\0';
	editedhost[sizeof editedhost - 1] = '\0';
}

static char *putlocation;

	void
putstr(s)
	register char *s;
{

	while (*s)
		putchr(*s++);
}

	void
putchr(cc)
	int cc;
{
	*putlocation++ = cc;
}

/*
 * This is split on two lines so that SCCS will not see the M
 * between two % signs and expand it...
 */
static char fmtstr[] = { "%l:%M\
%P on %A, %d %B %Y" };

	void
putf(cp, where)
	register char *cp;
	char *where;
{
	char *slash;
	time_t t;
	char db[100];
#ifdef HAVE_UNAME
        struct utsname u;
#endif /* HAVE_UNAME */
#ifdef	STREAMSPTY
#ifndef strchr
	extern char *strchr();
#endif
#else
#ifndef strrchr
	extern char *strrchr();
#endif
#endif

	putlocation = where;

	while (*cp) {
		if (*cp != '%') {
			putchr(*cp++);
			continue;
		}
		switch (*++cp) {

		case 't':
#ifdef	STREAMSPTY
			/* names are like /dev/pts/2 -- we want pts/2 */
			slash = strchr(line+1, '/');
#else
			slash = strrchr(line, '/');
#endif
			if (slash == (char *) 0)
				putstr(line);
			else
				putstr(&slash[1]);
			break;

		case 'h':
			putstr(editedhost);
			break;

		case 'd':
			(void)time(&t);
			(void)strftime(db, sizeof(db), fmtstr, localtime(&t));
			putstr(db);
			break;

#ifdef HAVE_UNAME
		case 's':
			if (uname(&u) == 0)
			    putstr(u.sysname);
		        break;
		case 'n':
			if (uname(&u) == 0)
			    putstr(u.nodename);
		        break;
		case 'r':
			if (uname(&u) == 0)
			    putstr(u.release);
		        break;
		case 'v':
			if (uname(&u) == 0)
			    putstr(u.version);
		        break;
		case 'm':
			if (uname(&u) == 0)
			    putstr(u.machine);
		        break;
#endif /* HAVE_UNAME */
		case '%':
			putchr('%');
			break;
		}
		cp++;
	}
}

#ifdef DIAGNOSTICS
/*
 * Print telnet options and commands in plain text, if possible.
 */

static FILE *diag_file = NULL;
static char diag_file_name[] = "/tmp/telnetd-diag-XXXXXX";

void diag_dup(int fd)
{
    int dffd = dup(fd);
    if ( dffd != -1 )
	diag_file = fdopen(dffd, "w");
    if (diag_file)
	diagnostic |= TD_TO_FILE;
}

void diag_init_file(void)
{
    int dffd = mkstemp(diag_file_name);
    if (dffd != -1) {
	/* change file mode so it's readable for everyone. maybe that's
	 * not what everybody wants, but I want it like that now.
	 */
	fchmod(dffd, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
	diag_file = fdopen(dffd, "w");
	if (diag_file) {
	    diagnostic |= TD_TO_FILE;
	    net_printf("\r\nDiagnostic output to file %s\r\n", diag_file_name);
	    nfrontp += strlen(nfrontp);
	}
    }
}

#ifdef __STDC__
int diag_printf(const char *fmt, ...)
#else
int diag_printf(fmt, va_alist)
    char *fmt;
    va_dcl
#endif
{
    va_list ap;
#ifdef __STDC__
    va_start(ap, fmt);
#else
    va_start(ap);
#endif

    if (diag_file) {
	vfprintf(diag_file, fmt, ap);
	return 0;
    } else {
	int rv = net_vprintf(fmt, ap);
	nfrontp += rv;
	return rv;
    }
}

	void
printoption(fmt, option)
	register char *fmt;
	register int option;
{
	if (TELOPT_OK(option))
		diag_printf("%s %s\r\n", fmt, TELOPT(option));
	else if (TELCMD_OK(option))
		diag_printf("%s %s\r\n", fmt, TELCMD(option));
	else
		diag_printf("%s %d\r\n", fmt, option);
	return;
}

    void
printsub(direction, pointer, length)
    char		direction;	/* '<' or '>' */
    unsigned char	*pointer;	/* where suboption data sits */
    int			length;		/* length of suboption data */
{
    register int i;
    char buf[512];

        if (!(diagnostic & TD_OPTIONS))
		return;

	if (direction) {
	    diag_printf("td: %s suboption ",
					direction == '<' ? "recv" : "send");
	    if (length >= 3) {
		register int j;

		i = pointer[length-2];
		j = pointer[length-1];

		if (i != IAC || j != SE) {
		    diag_printf("(terminated by ");
		    if (TELOPT_OK(i))
			diag_printf("%s ", TELOPT(i));
		    else if (TELCMD_OK(i))
			diag_printf("%s ", TELCMD(i));
		    else
			diag_printf("%d ", i);
		    if (TELOPT_OK(j))
			diag_printf("%s", TELOPT(j));
		    else if (TELCMD_OK(j))
			diag_printf("%s", TELCMD(j));
		    else
			diag_printf("%d", j);
		    diag_printf(", not IAC SE!) ");
		}
	    }
	    length -= 2;
	}
	if (length < 1) {
	    diag_printf("(Empty suboption??\?)");
	    return;
	}
	switch (pointer[0]) {
	case TELOPT_TTYPE:
	    diag_printf("TERMINAL-TYPE ");
	    switch (pointer[1]) {
	    case TELQUAL_IS:
		diag_printf("IS \"%.*s\"", length-2, (char *)pointer+2);
		break;
	    case TELQUAL_SEND:
		diag_printf("SEND");
		break;
	    default:
		diag_printf("- unknown qualifier %d (0x%x).",
				pointer[1], pointer[1]);
	    }
	    break;
	case TELOPT_TSPEED:
	    diag_printf("TERMINAL-SPEED");
	    if (length < 2) {
		diag_printf(" (empty suboption??\?)");
		break;
	    }
	    switch (pointer[1]) {
	    case TELQUAL_IS:
		diag_printf(" IS %.*s", length-2, (char *)pointer+2);
		break;
	    default:
		if (pointer[1] == 1)
		    diag_printf(" SEND");
		else
		    diag_printf(" %d (unknown)", pointer[1]);
		for (i = 2; i < length; i++) {
		    diag_printf(" ?%d?", pointer[i]);
		}
		break;
	    }
	    break;

	case TELOPT_LFLOW:
	    diag_printf("TOGGLE-FLOW-CONTROL");
	    if (length < 2) {
		diag_printf(" (empty suboption??\?)");
		break;
	    }
	    switch (pointer[1]) {
	    case LFLOW_OFF:
		diag_printf(" OFF"); break;
	    case LFLOW_ON:
		diag_printf(" ON"); break;
	    case LFLOW_RESTART_ANY:
		diag_printf(" RESTART-ANY"); break;
	    case LFLOW_RESTART_XON:
		diag_printf(" RESTART-XON"); break;
	    default:
		diag_printf(" %d (unknown)", pointer[1]);
	    }
	    for (i = 2; i < length; i++) {
		diag_printf(" ?%d?", pointer[i]);
	    }
	    break;

	case TELOPT_NAWS:
	    diag_printf("NAWS");
	    if (length < 2) {
		diag_printf(" (empty suboption??\?)");
		break;
	    }
	    if (length == 2) {
		diag_printf(" ?%d?", pointer[1]);
		break;
	    }
	    diag_printf(" %d %d (%d)",
		pointer[1], pointer[2],
		(int)((((unsigned int)pointer[1])<<8)|((unsigned int)pointer[2])));
	    if (length == 4) {
		diag_printf(" ?%d?", pointer[3]);
		break;
	    }
	    diag_printf(" %d %d (%d)",
		pointer[3], pointer[4],
		(int)((((unsigned int)pointer[3])<<8)|((unsigned int)pointer[4])));
	    for (i = 5; i < length; i++) {
		diag_printf(" ?%d?", pointer[i]);
	    }
	    break;

	case TELOPT_LINEMODE:
	    diag_printf("LINEMODE ");
	    if (length < 2) {
		diag_printf(" (empty suboption??\?)");
		break;
	    }
	    switch (pointer[1]) {
	    case WILL:
		diag_printf("WILL ");
		goto common;
	    case WONT:
		diag_printf("WONT ");
		goto common;
	    case DO:
		diag_printf("DO ");
		goto common;
	    case DONT:
		diag_printf("DONT ");
	    common:
		if (length < 3) {
		    diag_printf("(no option??\?)");
		    break;
		}
		switch (pointer[2]) {
		case LM_FORWARDMASK:
		    diag_printf("Forward Mask");
		    for (i = 3; i < length; i++) {
			diag_printf(" %x", pointer[i]);
		    }
		    break;
		default:
		    diag_printf("%d (unknown)", pointer[2]);
		    for (i = 3; i < length; i++) {
			diag_printf(" %d", pointer[i]);
		    }
		    break;
		}
		break;
		
	    case LM_SLC:
		diag_printf("SLC");
		for (i = 2; i < length - 2; i += 3) {
		    if (SLC_NAME_OK(pointer[i+SLC_FUNC]))
			diag_printf(" %s", SLC_NAME(pointer[i+SLC_FUNC]));
		    else
			diag_printf(" %d", pointer[i+SLC_FUNC]);
		    switch (pointer[i+SLC_FLAGS]&SLC_LEVELBITS) {
		    case SLC_NOSUPPORT:
			diag_printf(" NOSUPPORT"); break;
		    case SLC_CANTCHANGE:
			diag_printf(" CANTCHANGE"); break;
		    case SLC_VARIABLE:
			diag_printf(" VARIABLE"); break;
		    case SLC_DEFAULT:
			diag_printf(" DEFAULT"); break;
		    }
		    diag_printf("%s%s%s",
			pointer[i+SLC_FLAGS]&SLC_ACK ? "|ACK" : "",
			pointer[i+SLC_FLAGS]&SLC_FLUSHIN ? "|FLUSHIN" : "",
			pointer[i+SLC_FLAGS]&SLC_FLUSHOUT ? "|FLUSHOUT" : "");
		    if (pointer[i+SLC_FLAGS]& ~(SLC_ACK|SLC_FLUSHIN|
						SLC_FLUSHOUT| SLC_LEVELBITS)) {
			diag_printf("(0x%x)", pointer[i+SLC_FLAGS]);
		    }
		    diag_printf(" %d;", pointer[i+SLC_VALUE]);
		    if ((pointer[i+SLC_VALUE] == IAC) &&
			(pointer[i+SLC_VALUE+1] == IAC))
				i++;
		}
		for (; i < length; i++) {
		    diag_printf(" ?%d?", pointer[i]);
		}
		break;

	    case LM_MODE:
		diag_printf("MODE ");
		if (length < 3) {
		    diag_printf("(no mode??\?)");
		    break;
		}
		{
		    char tbuf[32];
		    snprintf(tbuf, sizeof(tbuf), "%s%s%s%s%s",
			pointer[2]&MODE_EDIT ? "|EDIT" : "",
			pointer[2]&MODE_TRAPSIG ? "|TRAPSIG" : "",
			pointer[2]&MODE_SOFT_TAB ? "|SOFT_TAB" : "",
			pointer[2]&MODE_LIT_ECHO ? "|LIT_ECHO" : "",
			pointer[2]&MODE_ACK ? "|ACK" : "");
		    diag_printf("%s", tbuf[1] ? &tbuf[1] : "0");
		}
		if (pointer[2]&~(MODE_EDIT|MODE_TRAPSIG|MODE_ACK)) {
		    diag_printf(" (0x%x)", pointer[2]);
		}
		for (i = 3; i < length; i++) {
		    diag_printf(" ?0x%x?", pointer[i]);
		}
		break;
	    default:
		diag_printf("%d (unknown)", pointer[1]);
		for (i = 2; i < length; i++) {
		    diag_printf(" %d", pointer[i]);
		}
	    }
	    break;

	case TELOPT_STATUS: {
	    register char *cp;
	    register int j, k;

	    diag_printf("STATUS");

	    switch (pointer[1]) {
	    default:
		if (pointer[1] == TELQUAL_SEND)
		    diag_printf(" SEND");
		else
		    diag_printf(" %d (unknown)", pointer[1]);
		for (i = 2; i < length; i++) {
		    diag_printf(" ?%d?", pointer[i]);
		}
		break;
	    case TELQUAL_IS:
		diag_printf(" IS\r\n");

		for (i = 2; i < length; i++) {
		    switch(pointer[i]) {
		    case DO:	cp = "DO"; goto common2;
		    case DONT:	cp = "DONT"; goto common2;
		    case WILL:	cp = "WILL"; goto common2;
		    case WONT:	cp = "WONT"; goto common2;
		    common2:
			i++;
			if (TELOPT_OK(pointer[i]))
			    diag_printf(" %s %s", cp, TELOPT(pointer[i]));
			else
			    diag_printf(" %s %d", cp, pointer[i]);

			diag_printf("\r\n");
			break;

		    case SB:
			diag_printf(" SB ");
			i++;
			j = k = i;
			while (j < length) {
			    if (pointer[j] == SE) {
				if (j+1 == length)
				    break;
				if (pointer[j+1] == SE)
				    j++;
				else
				    break;
			    }
			    pointer[k++] = pointer[j++];
			}
			printsub(0, &pointer[i], k - i);
			if (i < length) {
			    diag_printf(" SE");
			    i = j;
			} else
			    i = j - 1;

			diag_printf("\r\n");
			break;

		    default:
			diag_printf(" %d", pointer[i]);
			break;
		    }
		}
		break;
	    }
	    break;
	  }

	case TELOPT_XDISPLOC:
	    diag_printf("X-DISPLAY-LOCATION ");
	    switch (pointer[1]) {
	    case TELQUAL_IS:
		diag_printf("IS \"%.*s\"", length-2, (char *)pointer+2);
		break;
	    case TELQUAL_SEND:
		diag_printf("SEND");
		break;
	    default:
		diag_printf("- unknown qualifier %d (0x%x).",
				pointer[1], pointer[1]);
	    }
	    break;

	case TELOPT_NEW_ENVIRON:
	    diag_printf("NEW-ENVIRON ");
	    goto env_common1;
	case TELOPT_OLD_ENVIRON:
	    diag_printf("OLD-ENVIRON");
	env_common1:
	    switch (pointer[1]) {
	    case TELQUAL_IS:
		diag_printf("IS ");
		goto env_common;
	    case TELQUAL_SEND:
		diag_printf("SEND ");
		goto env_common;
	    case TELQUAL_INFO:
		diag_printf("INFO ");
	    env_common:
		{
		    register int noquote = 2;
		    for (i = 2; i < length; i++ ) {
			switch (pointer[i]) {
			case NEW_ENV_VAR:
			    diag_printf("\" VAR " + noquote);
			    noquote = 2;
			    break;

			case NEW_ENV_VALUE:
			    diag_printf("\" VALUE " + noquote);
			    noquote = 2;
			    break;

			case ENV_ESC:
			    diag_printf("\" ESC " + noquote);
			    noquote = 2;
			    break;

			case ENV_USERVAR:
			    diag_printf("\" USERVAR " + noquote);
			    noquote = 2;
			    break;

			default:
			def_case:
			    if (isprint(pointer[i]) && pointer[i] != '"') {
				if (noquote) {
				    diag_printf("\"");
				    noquote = 0;
				}
				diag_printf("%c", pointer[i]);
			    } else {
				diag_printf("\" %03o " + noquote,
							pointer[i]);
				noquote = 2;
			    }
			    break;
			}
		    }
		    if (!noquote)
			 diag_printf("\"");
		    break;
		}
	    }
	    break;

#if	defined(AUTHENTICATION)
	case TELOPT_AUTHENTICATION:
	    diag_printf("AUTHENTICATION");

	    if (length < 2) {
		diag_printf(" (empty suboption??\?)");
		break;
	    }
	    switch (pointer[1]) {
	    case TELQUAL_REPLY:
	    case TELQUAL_IS:
		diag_printf(" %s ", (pointer[1] == TELQUAL_IS) ?
							"IS" : "REPLY");
		if (AUTHTYPE_NAME_OK(pointer[2]))
		    diag_printf("%s ", AUTHTYPE_NAME(pointer[2]));
		else
		    diag_printf("%d ", pointer[2]);
		if (length < 3) {
		    diag_printf("(partial suboption??\?)");
		    break;
		}
		diag_printf("%s|%s",
			((pointer[3] & AUTH_WHO_MASK) == AUTH_WHO_CLIENT) ?
			"CLIENT" : "SERVER",
			((pointer[3] & AUTH_HOW_MASK) == AUTH_HOW_MUTUAL) ?
			"MUTUAL" : "ONE-WAY");

		auth_printsub(&pointer[1], length - 1, buf, sizeof(buf));
		diag_printf("%s", buf);
		break;

	    case TELQUAL_SEND:
		i = 2;
		diag_printf(" SEND ");
		while (i < length) {
		    if (AUTHTYPE_NAME_OK(pointer[i]))
			diag_printf("%s ", AUTHTYPE_NAME(pointer[i]));
		    else
			diag_printf("%d ", pointer[i]);
		    if (++i >= length) {
			diag_printf("(partial suboption??\?)");
			break;
		    }
		    diag_printf("%s|%s ",
			((pointer[i] & AUTH_WHO_MASK) == AUTH_WHO_CLIENT) ?
							"CLIENT" : "SERVER",
			((pointer[i] & AUTH_HOW_MASK) == AUTH_HOW_MUTUAL) ?
							"MUTUAL" : "ONE-WAY");
		    ++i;
		}
		break;

	    case TELQUAL_NAME:
		i = 2;
		diag_printf(" NAME \"");
		while (i < length) {
		    diag_printf("%c", pointer[i++]);
		}
		diag_printf("\"");
		break;

	    default:
		    for (i = 2; i < length; i++) {
			diag_printf(" ?%d?", pointer[i]);
		    }
		    break;
	    }
	    break;
#endif

#ifdef	ENCRYPTION
	case TELOPT_ENCRYPT:
	    diag_printf("ENCRYPT");
	    if (length < 2) {
		diag_printf(" (empty suboption?)");
		break;
	    }
	    switch (pointer[1]) {
	    case ENCRYPT_START:
		diag_printf(" START");
		break;

	    case ENCRYPT_END:
		diag_printf(" END");
		break;

	    case ENCRYPT_REQSTART:
		diag_printf(" REQUEST-START");
		break;

	    case ENCRYPT_REQEND:
		diag_printf(" REQUEST-END");
		break;

	    case ENCRYPT_IS:
	    case ENCRYPT_REPLY:
		diag_printf(" %s ", (pointer[1] == ENCRYPT_IS) ?
							"IS" : "REPLY");
		if (length < 3) {
		    diag_printf(" (partial suboption??\?)");
		    break;
		}
		if (ENCTYPE_NAME_OK(pointer[2]))
		    diag_printf("%s ", ENCTYPE_NAME(pointer[2]));
		else
		    diag_printf(" %d (unknown)", pointer[2]);

		encrypt_printsub(&pointer[1], length - 1, buf, sizeof(buf));
		diag_printf("%s", buf);
		break;

	    case ENCRYPT_SUPPORT:
		i = 2;
		diag_printf(" SUPPORT ");
		while (i < length) {
		    if (ENCTYPE_NAME_OK(pointer[i]))
			diag_printf("%s ", ENCTYPE_NAME(pointer[i]));
		    else
			diag_printf("%d ", pointer[i]);
		    i++;
		}
		break;

	    case ENCRYPT_ENC_KEYID:
		diag_printf(" ENC_KEYID", pointer[1]);
		goto encommon;

	    case ENCRYPT_DEC_KEYID:
		diag_printf(" DEC_KEYID", pointer[1]);
		goto encommon;

	    default:
		diag_printf(" %d (unknown)", pointer[1]);
	    encommon:
		for (i = 2; i < length; i++) {
		    diag_printf(" %d", pointer[i]);
		}
		break;
	    }
	    break;
#endif	/* ENCRYPTION */

#ifdef TLS
	case TELOPT_START_TLS:
	    diag_printf("START-TLS ");
	    switch (pointer[1]) {
	    case TLS_FOLLOWS:
		diag_printf("FOLLOWS");
		break;
	    default:
		diag_printf("- unknown qualifier %d (0x%x).",
				pointer[1], pointer[1]);
	    }
	    break;
#endif /* TLS */
#ifdef FWD_X
	case TELOPT_FORWARD_X:
	    diag_printf("FORWARD-X ");
	    switch (pointer[1]) {
	    case FWDX_SCREEN:
		diag_printf("SCREEN %d", pointer[2]);
		break;
	    case FWDX_OPEN:
		diag_printf("OPEN %d %d", pointer[2], pointer[3]);
		break;
	    case FWDX_CLOSE:
		diag_printf("CLOSE %d %d", pointer[2], pointer[3]);
		break;
	    case FWDX_DATA:
		diag_printf("DATA %d %d [%d bytes]", pointer[2], pointer[3], length - 4);
		break;
	    case FWDX_OPTIONS: {
	    	int n;
		diag_printf("OPTIONS");
		for (n = 2; n < length; n++) {
		    diag_printf(" %d", pointer[n]);
		    }
		break;
		}
	    default:
		diag_printf("- unknown qualifier %d (0x%x).",
				pointer[1], pointer[1]);
	    }
	    break;
#endif /* FWD_X */

	default:
	    if (TELOPT_OK(pointer[0]))
	        diag_printf("%s (unknown)", TELOPT(pointer[0]));
	    else
	        diag_printf("%d (unknown)", pointer[0]);
	    for (i = 1; i < length; i++)
		diag_printf(" %d", pointer[i]);
	    break;
	}
	diag_printf("\r\n");
}

/*
 * Dump a data buffer in hex and ascii to the output data stream.
 */
	void
printdata(tag, ptr, cnt)
	register char *tag;
	register const unsigned char *ptr;
	register int cnt;
{
	register int i;
	char xbuf[30];

	while (cnt) {
		/* flush net output buffer if no room for new data */
		if (!(diagnostic & TD_TO_FILE) && (&netobuf[netobuf_size] - nfrontp) < 80) {
			netflush();
		}

		/* add a line of output */
		/* some "tag"s are 3 chars, not two */
		if (tag[2])
			diag_printf("%s ", tag);
		else
			diag_printf("%s: ", tag);
		for (i = 0; i < 20 && cnt; i++) {
			diag_printf("%02x", *ptr);
			if (isprint(*ptr)) {
				xbuf[i] = *ptr;
			} else {
				xbuf[i] = '.';
			}
			if (i % 2) {
				diag_printf(" ");
			}
			cnt--;
			ptr++;
		}
		xbuf[i] = '\0';
		/* we want to adjust the acsii part of the output to the right.
		 * how many spaces are needed? */
		i = 20 - i;
		i = i * 2 + i / 2 + ( i % 2 ? 1 : 0 );
		diag_printf("%*s %s\r\n", i, "", xbuf );
	} 
}
#endif /* DIAGNOSTICS */

int net_vprintf(const char *fmt, va_list ap)
{
    /* so, why don't I update "nfrontp" in this function? well, it's actually
     * hard to do, since there might be embedded '\0's in the data, and since
     * one can't rely on the return value from vsnprintf() on all platforms.
     * the caller of net_(v)printf() must take care of updating "nfrontp".
     * however, if there isn't any embedded '\0's, it's safe to do:
     * nfrontp += net_printf()	       				/Luna
     */
#define SNP_MAXBUF 1024000
    register int ret, netobuf_left, nfrontp_idx, nbackp_idx;
	
    /* here I boldly assume that snprintf() and vsnprintf() uses the same
     * return value convention. if not, what kind of libc is this? ;-)
     */
    nfrontp_idx = nfrontp - netobuf;	/* equal to how filled netobuf is */
    nbackp_idx = nbackp - netobuf;
    netobuf_left = netobuf_size - (nfrontp - netobuf);

    ret = vsnprintf(nfrontp, netobuf_left, fmt, ap);
#ifdef SNPRINTF_OK
    /* this one returns the number of bytes it wants to write in case of overflow */
    if (ret >= netobuf_left) {
	/* netobuf was too small, increase it */
	register int new_size = ret + 1 + nfrontp_idx;
	register char *p;
	if (new_size > SNP_MAXBUF)
	    syslog(LOG_ERR, "netobuf_size wanted to go beyond %d", SNP_MAXBUF);
	else {
	    p = realloc(netobuf, new_size);
	    if (p) {
		netobuf = p;
		nfrontp = p + nfrontp_idx;
		nbackp = p + nbackp_idx;
		netobuf_size = new_size;
		vsnprintf(nfrontp, ret + 1, fmt, ap);
	    }
	}
    }
#else
# ifdef SNPRINTF_HALFBROKEN
    /* this one returns the number of bytes written (excl. \0) in case of overflow */
#  define SNP_OVERFLOW(x, y) ( x == y ? 1 : 0 )
#  define SNP_NOERROR(x)     ( x < 0 ? 0 : 1 )
# else
#  ifdef SNPRINTF_BROKEN
    /* this one returns -1 in case of overflow */
#   define SNP_OVERFLOW(x, y) ( x < 0 ? 1 : 0 )
#   define SNP_NOERROR(x)     ( 1 )  /* if -1 means overflow, what's the error indication? */
#  else
#   error No valid SNPRINTF_... macro defined!
#  endif /* !SNPRINTF_BROKEN */
# endif /* !SNPRINTF_HALFBROKEN */
    if (SNP_NOERROR(ret) && SNP_OVERFLOW(ret, netobuf_left - 1)) {
	/* netobuf was too small, increase it */
	register int new_size = netobuf_size;
	register char *p;
	do {
	    if ((new_size *= 2) > SNP_MAXBUF) {	/* try to double the size */
		syslog(LOG_ERR, "netobuf_size wanted to go beyond %d", SNP_MAXBUF);
		break;
	    }
	    p = realloc(netobuf, new_size);
	    if (p) {
		netobuf = p;
		nfrontp = p + nfrontp_idx;
		nbackp = p + nbackp_idx;
		netobuf_size = new_size;
		netobuf_left = netobuf_size - (nfrontp - netobuf);
		ret = vsnprintf(nfrontp, netobuf_left, fmt, ap);
	    } else
		break;
	} while (SNP_NOERROR(ret) && SNP_OVERFLOW(ret, netobuf_left - 1));
    }
#endif /* !SNPRINTF_OK */
    return strlen(nfrontp);
}

#ifdef __STDC__
int net_printf(const char *fmt, ...)
#else
int net_printf(fmt, va_alist)
    char *fmt;
    va_dcl
#endif
{
    va_list ap;
#ifdef __STDC__
    va_start(ap, fmt);
#else
    va_start(ap);
#endif

    return net_vprintf(fmt, ap);
}
