/*
 * Copyright (c) 1999 - 2002 Peter 'Luna' Runestig <peter@runestig.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modifi-
 * cation, are permitted provided that the following conditions are met:
 *
 *   o  Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *
 *   o  Redistributions in binary form must reproduce the above copyright no-
 *      tice, this list of conditions and the following disclaimer in the do-
 *      cumentation and/or other materials provided with the distribution.
 *
 *   o  The names of the contributors may not be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LI-
 * ABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUEN-
 * TIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEV-
 * ER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABI-
 * LITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef lint
static char copyright[] =
"@(#) Copyright (c) Peter 'Luna' Runestig 1999 - 2002 <peter@runestig.com>.\n";
#endif /* not lint */

#if !defined(PR_OBSD_FTPD) && !defined(PR_PROFTPD) && !defined(PR_TELNETD)
#error You must define PR_OBSD_FTPD, PR_PROFTPD or PR_TELNETD !
#endif

#ifdef PR_PROFTPD
/* ProFTPD special includes */
#include "conf.h"
#include <privs.h>
#endif /* PR_PROFTPD */

#if defined(PR_OBSD_FTPD) || defined(PR_TELNETD)
#include <syslog.h>
#include <unistd.h>
#include <pwd.h>
#endif /* PR_OBSD_FTPD || PR_TELNETD */

#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/poll.h>
#include <openssl/ssl.h>
#include <openssl/x509v3.h>
#include <openssl/err.h>
#include <openssl/rand.h>

#ifdef TLS_SESSION_FILE_CACHE
#include <dirent.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <sys/stat.h>
#include <openssl/pem.h>
#endif /* TLS_SESSION_FILE_CACHE */

#ifdef PR_OBSD_FTPD
# ifdef __STDC__
#  include <stdarg.h>
# else
#  include <varargs.h>
# endif /* __STDC__ */
#endif /* PR_OBSD_FTPD */

#ifdef PR_PROFTPD
/* ProFTPD special includes */
#include "io.h"
#include "inet.h"
#endif /* PR_PROFTPD */

#ifdef PR_TELNETD
#include "telnetd.h"
extern int net; /* the socket */
#endif /* PR_TELNETD */

#include "tls_dh.h"

#if OPENSSL_VERSION_NUMBER < 0x00905100
/* ASN1_BIT_STRING_cmp was renamed in 0.9.5 */
#define M_ASN1_BIT_STRING_cmp ASN1_BIT_STRING_cmp
#endif

#if defined(PR_OBSD_FTPD) || defined(PR_PROFTPD)
#define DEFRSACERTFILE		"ftpd-rsa.pem"
#define DEFRSACERTCHAINFILE	"ftpd-rsa-chain.pem"
#define DEFRSAKEYFILE		"ftpd-rsa-key.pem" 
#define DEFDSACERTFILE		"ftpd-dsa.pem"
#define DEFDSACERTCHAINFILE	"ftpd-dsa-chain.pem"
#define DEFDSAKEYFILE		"ftpd-dsa-key.pem" 
#define DEFCRLFILE		"ftpd-crl.pem"
#define DEFDHPARAMFILE		"ftpd-dhparam.pem"
#elif defined(PR_TELNETD)
#define DEFRSACERTFILE		"telnetd-rsa.pem"
#define DEFRSACERTCHAINFILE	"telnetd-rsa-chain.pem"
#define DEFRSAKEYFILE		"telnetd-rsa-key.pem" 
#define DEFDSACERTFILE		"telnetd-dsa.pem"
#define DEFDSACERTCHAINFILE	"telnetd-dsa-chain.pem"
#define DEFDSAKEYFILE		"telnetd-dsa-key.pem" 
#define DEFCRLFILE		"telnetd-crl.pem"
#define DEFDHPARAMFILE		"telnetd-dhparam.pem"
#endif /* PR_TELNETD */
#define DEFAULTCIPHERLIST       "HIGH:MEDIUM:LOW:+KRB5:+ADH:+EXP"

#if defined(PR_OBSD_FTPD) || defined(PR_TELNETD)
# define SYSLOG	syslog
#elif defined(PR_PROFTPD)
# define SYSLOG	log_pri
#endif
/* define if you want to check for OpenSSL-related memory leaks */
/* #define DEBUG_OPENSSL_MEM */

#ifdef PR_OBSD_FTPD
typedef struct {
    SSL *ssl;
    int sock;
} CONN;

void	reply(int, const char *, ...);
void	dologout(int status);
#endif /* PR_OBSD_FTPD */

int	x509_to_user(X509 *peer_cert, char *userid, int len);

int	tls_no_verify = 0;
int	tls_dont_request_cert = 0;
int	tls_required = 0;
int	tls_client_cert_req = 0;
#ifdef ZLIB
int	tls_nozlib = 0;
#endif /* ZLIB */
char 	*tls_rsa_key_file = NULL;
char	*tls_rsa_cert_file = NULL;
char	*tls_rsa_cert_chain_file = NULL;
char 	*tls_dsa_key_file = NULL;
char	*tls_dsa_cert_file = NULL;
char	*tls_dsa_cert_chain_file = NULL;
char	*tls_crl_file = NULL;
char	*tls_crl_dir = NULL;
char	*tls_dhparam_file = NULL;
char	*tls_rand_file = NULL;
char	*tls_cipher_list = NULL;
static SSL_CTX	*ssl_ctx = NULL;
static X509_STORE *crl_store = NULL;
static DH *tmp_dh = NULL;
static RSA *tmp_rsa = NULL;

#ifdef PR_TELNETD
int 	tls_active = 0;
int 	tls_follows_from_client = 0;
static SSL *ssl = NULL;
#endif /* PR_TELNETD */

#ifdef PR_PROFTPD
int	tls_on_ctrl = 0;
int	tls_on_data = 0;
int 	tls_implicit = 0;
int	tls_init_error = 0;
static SSL *first_ssl = NULL;
#endif /* PR_PROFTPD */

#ifdef PR_OBSD_FTPD
int	tls_on_ctrl = 0;
int	tls_on_data = 0;
int	tls_pass_passthrough = 0;
CONN	data_conn = { NULL, -1 }, ctrl_conn = { NULL, -1 };

static SSL *SOCK_TO_SSL(int s)
{
    /* stdin/stdout needs special treatment since it's two different file
     * numbers reffering to the same socket
     */
    if (s == 0 || s == 1) {
	if (data_conn.sock == 0 || data_conn.sock == 1)
	    return data_conn.ssl;
	else if (ctrl_conn.sock == 0 || ctrl_conn.sock == 1)
	    return ctrl_conn.ssl;
	else
	    return NULL;
    } else
	return s == data_conn.sock ? data_conn.ssl :
	     ( s == ctrl_conn.sock ? ctrl_conn.ssl : NULL );
}
#endif /* PR_OBSD_FTPD */

static char *file_fullpath(char *fn)
{
    static char fp[256];
    FILE *file;
    char *dir;
    
    /* check if it is a full path already */
    if ((strchr(fn, '/'))) {
	if ((file = fopen(fn, "r"))) {
	    fclose(file);
	    return fn;
	} else
	    return NULL;
    }
    /* check if it is in current dir */
    if ((file = fopen(fn, "r"))) {
    	fclose(file);
	return fn;
    }
    if (!(dir = getenv(X509_get_default_cert_dir_env())))  /* $SSL_CERT_DIR */
    	dir = (char *) X509_get_default_cert_dir();
    snprintf(fp, sizeof(fp), "%s/%s", dir, fn);
    if ((file = fopen(fp, "r"))) {
    	fclose(file);
	return fp;
    }
    dir = (char *) X509_get_default_private_dir();
    snprintf(fp, sizeof(fp), "%s/%s", dir, fn);
    if ((file = fopen(fp, "r"))) {
    	fclose(file);
	return fp;
    }
    return NULL;
}

/* we need this so we don't mix static and malloc'ed strings */
void tls_set_defaults(void)
{
    if (!tls_rsa_key_file)
	tls_rsa_key_file = strdup(DEFRSAKEYFILE);
    if (!tls_rsa_cert_file)
	tls_rsa_cert_file = strdup(DEFRSACERTFILE);
    if (!tls_rsa_cert_chain_file)
	tls_rsa_cert_chain_file = strdup(DEFRSACERTCHAINFILE);
    if (!tls_dsa_key_file)
	tls_dsa_key_file = strdup(DEFDSAKEYFILE);
    if (!tls_dsa_cert_file)
	tls_dsa_cert_file = strdup(DEFDSACERTFILE);
    if (!tls_dsa_cert_chain_file)
	tls_dsa_cert_chain_file = strdup(DEFDSACERTCHAINFILE);
    if (!tls_crl_file)
	tls_crl_file = strdup(DEFCRLFILE);
    if (!tls_crl_dir && (tls_crl_dir = malloc(strlen(X509_get_default_cert_area()) + 5)))
    	sprintf(tls_crl_dir, "%s/crl", X509_get_default_cert_area());  /* safe */
    if (!tls_dhparam_file)
	tls_dhparam_file = strdup(DEFDHPARAMFILE);
    if (!tls_cipher_list)
	tls_cipher_list = strdup(DEFAULTCIPHERLIST);
}

#if defined(PR_OBSD_FTPD) || defined(PR_TELNETD)
int tls_optarg(char *optarg)
/* returns 0 if optarg OK, else 1 */
{
    char *p;

    if ((p = strchr(optarg, '='))) {
    	*p++ = 0;
	if (!strcmp(optarg, "cert") || !strcmp(optarg, "rsacert")) {
	    if (tls_rsa_cert_file)
	    	free(tls_rsa_cert_file);
	    tls_rsa_cert_file = strdup(p);
	} else if (!strcmp(optarg, "chain") || !strcmp(optarg, "rsachain")) {
	    if (tls_rsa_cert_chain_file)
	    	free(tls_rsa_cert_chain_file);
	    tls_rsa_cert_chain_file = strdup(p);
	} else if (!strcmp(optarg, "key") || !strcmp(optarg, "rsakey")) {
	    if (tls_rsa_key_file)
	    	free(tls_rsa_key_file);
	    tls_rsa_key_file = strdup(p);
	} else if (!strcmp(optarg, "dsacert")) {
	    if (tls_dsa_cert_file)
	    	free(tls_dsa_cert_file);
	    tls_dsa_cert_file = strdup(p);
	} else if (!strcmp(optarg, "dsachain")) {
	    if (tls_dsa_cert_chain_file)
	    	free(tls_dsa_cert_chain_file);
	    tls_dsa_cert_chain_file = strdup(p);
	} else if (!strcmp(optarg, "dsakey")) {
	    if (tls_dsa_key_file)
	    	free(tls_dsa_key_file);
	    tls_dsa_key_file = strdup(p);
	} else if (!strcmp(optarg, "dhparam")) {
	    if (tls_dhparam_file)
	    	free(tls_dhparam_file);
	    tls_dhparam_file = strdup(p);
	} else if (!strcmp(optarg, "crlfile")) {
	    if (tls_crl_file)
	    	free(tls_crl_file);
	    tls_crl_file = strdup(p);
	} else if (!strcmp(optarg, "crldir")) {
	    if (tls_crl_dir)
	    	free(tls_crl_dir);
	    tls_crl_dir = strdup(p);
	} else if (!strcmp(optarg, "cipher")) {
	    if (tls_cipher_list)
	    	free(tls_cipher_list);
	    tls_cipher_list = strdup(p);
	} else
	    return 1;
    } else if (!strcmp(optarg, "certsok"))
	tls_no_verify = 1;
    else if (!strcmp(optarg, "dontrequestcert"))
	tls_dont_request_cert = 1;
    else if (!strcmp(optarg, "required"))
	tls_required = 1;
    else if (!strcmp(optarg, "clientcertreq"))
	tls_client_cert_req = 1;
#ifdef ZLIB
    else if (!strcmp(optarg, "nozlib"))
    	tls_nozlib = 1;
#endif /* ZLIB */
    else
	return 1;

    return 0;
}
#endif /* PR_OBSD_FTPD || PR_TELNETD */

#ifdef PR_OBSD_FTPD
int tls_active(int s)
{
    if (SOCK_TO_SSL(s))
	return 1;
    else
	return 0;
}
#endif /* PR_OBSD_FTPD */

/* if we are using OpenSSL 0.9.6 or newer, we want to use X509_NAME_print_ex()
 * instead of X509_NAME_oneline().
 */
static char *x509_name_oneline(X509_NAME *n, char *buf, int len)
{
#if OPENSSL_VERSION_NUMBER < 0x000906000
    return X509_NAME_oneline(n, buf, len);
#else
    BIO *mem = BIO_new(BIO_s_mem());
    char *data = NULL;
    int data_len = 0, ok;
    
    ok = X509_NAME_print_ex(mem, n, 0, XN_FLAG_ONELINE);
    if (ok)
	data_len = BIO_get_mem_data(mem, &data);
    if (data) {
	/* the 'data' returned is not '\0' terminated */
	if (buf) {
	    memcpy(buf, data, data_len < len ? data_len : len);
	    buf[data_len < len ? data_len : len - 1] = 0;
	    BIO_free(mem);
	    return buf;
	} else {
	    char *b = malloc(data_len + 1);
	    if (b) {
		memcpy(b, data, data_len);
		b[data_len] = 0;
	    }
	    BIO_free(mem);
	    return b;
	}
    } else {
	BIO_free(mem);
	return NULL;
    }
#endif /* OPENSSL_VERSION_NUMBER >= 0x000906000 */
}

char *tls_get_subject_name(SSL *ssl)
{
    static char name[256];
    X509 *cert;

    if ((cert = SSL_get_peer_certificate(ssl))) {
	x509_name_oneline(X509_get_subject_name(cert), name, sizeof(name));
	X509_free(cert);
	return name;
    } else
	return NULL;
}

static DH *tmp_dh_cb(SSL *ssl, int is_export, int keylength)
{
    FILE *fp;

    if (!tmp_dh) {
    	/* first try any 'tls_dhparam_file', else use built-in dh params */
    	if (tls_dhparam_file && (fp = fopen(tls_dhparam_file, "r"))) {
	    tmp_dh = PEM_read_DHparams(fp, NULL, NULL, NULL);
	    fclose(fp);
	    if (tmp_dh)
	    	return tmp_dh;
	}
	switch (keylength) {
	    case 512:	return tmp_dh = get_dh512();
	    case 768:	return tmp_dh = get_dh768();
	    case 1024:	return tmp_dh = get_dh1024();
	    case 1536:	return tmp_dh = get_dh1536();
	    case 2048:	return tmp_dh = get_dh2048();
	    default:	return tmp_dh = get_dh1024();
	}
    }
    else
    	return tmp_dh;
}

static RSA *tmp_rsa_cb(SSL *ssl, int is_export, int keylength)
{
    if (!tmp_rsa)
	tmp_rsa = RSA_generate_key(keylength, RSA_F4, NULL, NULL);
    return tmp_rsa;
}

/* check_file() expands 'file' to an existing full path or NULL if not found */
static void check_file(char **file)
{
    char *p;
    
    if (*file) {
    	p = file_fullpath(*file);
	if (p == *file)	/* same pointer returned from file_fullpath() */
	    return;
	free(*file);
	if (p) {
	    *file = malloc(strlen(p) + 1);
	    strcpy(*file, p);
	}
	else
	    *file = NULL;
    }
}

/* this one is (very much!) based on work by Ralf S. Engelschall <rse@engelschall.com>.
 * comments by Ralf.
 */
static int verify_crl(int ok, X509_STORE_CTX *ctx)
{
    X509_OBJECT obj;
    X509_NAME *subject;
    X509_NAME *issuer;
    X509 *xs;
    X509_CRL *crl;
    X509_REVOKED *revoked;
    X509_STORE_CTX store_ctx;
    long serial;
    int i, n, rc;
    char *cp;

    /*
     * Unless a revocation store for CRLs was created we
     * cannot do any CRL-based verification, of course.
     */
    if (!crl_store)
        return ok;

    /*
     * Determine certificate ingredients in advance
     */
    xs      = X509_STORE_CTX_get_current_cert(ctx);
    subject = X509_get_subject_name(xs);
    issuer  = X509_get_issuer_name(xs);

    /*
     * OpenSSL provides the general mechanism to deal with CRLs but does not
     * use them automatically when verifying certificates, so we do it
     * explicitly here. We will check the CRL for the currently checked
     * certificate, if there is such a CRL in the store.
     *
     * We come through this procedure for each certificate in the certificate
     * chain, starting with the root-CA's certificate. At each step we've to
     * both verify the signature on the CRL (to make sure it's a valid CRL)
     * and it's revocation list (to make sure the current certificate isn't
     * revoked).  But because to check the signature on the CRL we need the
     * public key of the issuing CA certificate (which was already processed
     * one round before), we've a little problem. But we can both solve it and
     * at the same time optimize the processing by using the following
     * verification scheme (idea and code snippets borrowed from the GLOBUS
     * project):
     *
     * 1. We'll check the signature of a CRL in each step when we find a CRL
     *    through the _subject_ name of the current certificate. This CRL
     *    itself will be needed the first time in the next round, of course.
     *    But we do the signature processing one round before this where the
     *    public key of the CA is available.
     *
     * 2. We'll check the revocation list of a CRL in each step when
     *    we find a CRL through the _issuer_ name of the current certificate.
     *    This CRLs signature was then already verified one round before.
     *
     * This verification scheme allows a CA to revoke its own certificate as
     * well, of course.
     */

    /*
     * Try to retrieve a CRL corresponding to the _subject_ of
     * the current certificate in order to verify it's integrity.
     */
    memset((char *)&obj, 0, sizeof(obj));
    X509_STORE_CTX_init(&store_ctx, crl_store, NULL, NULL);
    rc = X509_STORE_get_by_subject(&store_ctx, X509_LU_CRL, subject, &obj);
    X509_STORE_CTX_cleanup(&store_ctx);
    crl = obj.data.crl;
    if (rc > 0 && crl != NULL) {
        /*
         * Verify the signature on this CRL
         */
        if (X509_CRL_verify(crl, X509_get_pubkey(xs)) <= 0) {
            SYSLOG(LOG_ERR, "Invalid signature on CRL!");
            X509_STORE_CTX_set_error(ctx, X509_V_ERR_CRL_SIGNATURE_FAILURE);
            X509_OBJECT_free_contents(&obj);
            return 0;
        }

        /*
         * Check date of CRL to make sure it's not expired
         */
        i = X509_cmp_current_time(X509_CRL_get_nextUpdate(crl));
        if (i == 0) {
            SYSLOG(LOG_ERR, "Found CRL has invalid nextUpdate field.");
            X509_STORE_CTX_set_error(ctx, X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD);
            X509_OBJECT_free_contents(&obj);
            return 0;
        }
        if (i < 0) {
            SYSLOG(LOG_ERR,
		"Found CRL is expired - revoking all certificates until you get updated CRL.");
            X509_STORE_CTX_set_error(ctx, X509_V_ERR_CRL_HAS_EXPIRED);
            X509_OBJECT_free_contents(&obj);
            return 0;
        }
        X509_OBJECT_free_contents(&obj);
    }

    /*
     * Try to retrieve a CRL corresponding to the _issuer_ of
     * the current certificate in order to check for revocation.
     */
    memset((char *)&obj, 0, sizeof(obj));
    X509_STORE_CTX_init(&store_ctx, crl_store, NULL, NULL);
    rc = X509_STORE_get_by_subject(&store_ctx, X509_LU_CRL, issuer, &obj);
    X509_STORE_CTX_cleanup(&store_ctx);
    crl = obj.data.crl;
    if (rc > 0 && crl != NULL) {
        /*
         * Check if the current certificate is revoked by this CRL
         */
        n = sk_X509_REVOKED_num(X509_CRL_get_REVOKED(crl));
        for (i = 0; i < n; i++) {
            revoked = sk_X509_REVOKED_value(X509_CRL_get_REVOKED(crl), i);
            if (ASN1_INTEGER_cmp(revoked->serialNumber, X509_get_serialNumber(xs)) == 0) {

                serial = ASN1_INTEGER_get(revoked->serialNumber);
                cp = x509_name_oneline(issuer, NULL, 0);
                SYSLOG(LOG_ERR,
		    "Certificate with serial %ld (0x%lX) revoked per CRL from issuer %s",
		    serial, serial, cp ? cp : "(ERROR)");
                if (cp) free(cp);

                X509_STORE_CTX_set_error(ctx, X509_V_ERR_CERT_REVOKED);
                X509_OBJECT_free_contents(&obj);
                return 0;
            }
        }
        X509_OBJECT_free_contents(&obj);
    }
    return ok;
}

static int verify_callback(int ok, X509_STORE_CTX *ctx)
{
/*    SYSLOG(LOG_ERR, "depth = %d, error = %d, ok = %d\n", ctx->error_depth, ctx->error, ok);*/
/* TODO: Make up my mind on what to accept or not. Also what to syslog. */
/* TODO: The client has a little different verify_callback(), should it be
 *       like that here too? */
    /* we can configure the server to skip the peer's cert verification */
    if (tls_no_verify)
    	return 1;
    ok = verify_crl(ok, ctx);
    if (!ok) {
    	switch (ctx->error) {
	    case X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT:
	    	SYSLOG(LOG_ERR, "Error: Client's certificate is self signed.");
		ok = 0;
		break;
	    case X509_V_ERR_CERT_HAS_EXPIRED:
	    	SYSLOG(LOG_ERR, "Error: Client's certificate has expired.");
		ok = 0;
		break;
	    case X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY:
	    	SYSLOG(LOG_ERR,
		    "Error: Client's certificate issuer's certificate isn't available locally.");
		ok = 0;
		break;
	    case X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE:
	    	SYSLOG(LOG_ERR, "Error: Unable to verify leaf signature.");
		ok = 0;
		break;
	    case X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN:
		/* XXX this is strange. we get this error for certain clients (ie Jeff's
		 * K95) when all is ok. I think it's because the client is actually sending
		 * the whole CA cert. this must be figured out, but we let it pass for now.
		 * if the CA cert isn't available locally, we will fail anyway.
		 */
	    	SYSLOG(LOG_NOTICE, "Warning: Self signed certificate in chain.");
		ok = 1;
		break;
	    case X509_V_ERR_CERT_REVOKED:
	    	SYSLOG(LOG_ERR, "Error: Certificate revoked.");
		ok = 0;
		break;
	    default:
	    	SYSLOG(LOG_ERR,
		    "Error %d while verifying the client's certificate.", ctx->error);
		ok = 0;
	    	break;
	}
    }
    return ok;
}

static int seed_PRNG(void)
{
    char stackdata[1024];
    static char rand_file[200];
    FILE *fh;
    
#if OPENSSL_VERSION_NUMBER >= 0x00905100
    if (RAND_status())
	return 0;     /* PRNG already good seeded */
#endif
    /* if the device '/dev/urandom' is present, OpenSSL uses it by default.
     * check if it's present, else we have to make random data ourselfs.
     */
    if ((fh = fopen("/dev/urandom", "r"))) {
	fclose(fh);
	return 0;
    }
    /* the rand file is (openssl-dir)/.rnd */
    snprintf(rand_file, sizeof(rand_file), "%s/.rnd", X509_get_default_cert_area());
    tls_rand_file = rand_file;
    if (!RAND_load_file(rand_file, 1024)) {
	/* no .rnd file found, create new seed */
	unsigned int c;
	c = time(NULL);
	RAND_seed(&c, sizeof(c));
	c = getpid();
	RAND_seed(&c, sizeof(c));
	RAND_seed(stackdata, sizeof(stackdata));
    }
#if OPENSSL_VERSION_NUMBER >= 0x00905100
    if (!RAND_status())
	return 2;   /* PRNG still badly seeded */
#endif
    return 0;
}

#ifdef TLS_SESSION_FILE_CACHE
#define TLS_SFC_MAX_FILES	100

static char *peer_ip_as_string(int sock)
{
#ifdef INET6
    struct sockaddr_storage saddr;
#else
    struct sockaddr_in saddr;
#endif /* !INET6 */
    int saddr_len = sizeof(saddr);
    static char rv[50];

    if (getpeername(sock, (struct sockaddr *) &saddr, &saddr_len) != 0)
	return NULL;
    switch (((struct sockaddr *)&saddr)->sa_family) {
	case AF_INET:
	    snprintf(rv, sizeof(rv), "%08X",
		     htonl(((struct sockaddr_in *)&saddr)->sin_addr.s_addr));
	    break;
#ifdef INET6
	case AF_INET6:
	    snprintf(rv, sizeof(rv),
		"%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X",
		     ((struct sockaddr_in6 *)&saddr)->sin6_addr.s6_addr[0],
		     ((struct sockaddr_in6 *)&saddr)->sin6_addr.s6_addr[1],
		     ((struct sockaddr_in6 *)&saddr)->sin6_addr.s6_addr[2],
		     ((struct sockaddr_in6 *)&saddr)->sin6_addr.s6_addr[3],
		     ((struct sockaddr_in6 *)&saddr)->sin6_addr.s6_addr[4],
		     ((struct sockaddr_in6 *)&saddr)->sin6_addr.s6_addr[5],
		     ((struct sockaddr_in6 *)&saddr)->sin6_addr.s6_addr[6],
		     ((struct sockaddr_in6 *)&saddr)->sin6_addr.s6_addr[7],
		     ((struct sockaddr_in6 *)&saddr)->sin6_addr.s6_addr[8],
		     ((struct sockaddr_in6 *)&saddr)->sin6_addr.s6_addr[9],
		     ((struct sockaddr_in6 *)&saddr)->sin6_addr.s6_addr[10],
		     ((struct sockaddr_in6 *)&saddr)->sin6_addr.s6_addr[11],
		     ((struct sockaddr_in6 *)&saddr)->sin6_addr.s6_addr[12],
		     ((struct sockaddr_in6 *)&saddr)->sin6_addr.s6_addr[13],
		     ((struct sockaddr_in6 *)&saddr)->sin6_addr.s6_addr[14],
		     ((struct sockaddr_in6 *)&saddr)->sin6_addr.s6_addr[15]);
	    break;
#endif /* INET6 */
	default:
	    return NULL;
    }
    return rv;
}

FILE *make_sfc_file(void)
{
    DIR *dir;
    char filename[MAXPATHLEN], *peer;
    int n, fd;

    dir = opendir(TLS_SFC_DIR);
    /* if the TLS_SFC_DIR dir doesn't exist, we consider this function disabled */
    if (dir == NULL)
	return NULL;
    closedir(dir);
    /* the file name is based on the hexadecimal representation of the peer's
     * ip address, with `oss' (from `OpenSsl Session') as a prefix and a index
     * number as a postfix.
     * example: connected to 127.0.0.1 -> `oss7F000001.1'
     */
    if ((peer = peer_ip_as_string(net)) == NULL)
	return NULL;
    /* try to create a file withing the range of TLS_SFC_MAX_FILES */
    for (n = 1; n <= TLS_SFC_MAX_FILES; n++) {
	snprintf(filename, sizeof(filename), "%s/oss%s.%d", TLS_SFC_DIR, peer, n);
	/* fopen() doesn't seem to be able to do this atomically, but
	 * this open() call fails if `filename' exist.
	 */
	fd = open(filename, O_WRONLY | O_CREAT | O_EXCL, S_IRUSR | S_IWUSR);
	if (fd < 0) {
	    if (errno != EEXIST)
		return NULL;	     /* some other error */
	} else
	    return fdopen(fd, "w");  /* success! */
    }
    return NULL;
}

int session_timed_out(SSL_SESSION *s)
{
    if (SSL_SESSION_get_time(s) + SSL_SESSION_get_timeout(s) < time(NULL))
	return 1;
    else
	return 0;
}

int tls_sfc_server_load(SSL_CTX *sc)
{
    int match_len;
    FILE *file;
    DIR *dir;
    struct dirent *de;
    char filename[MAXPATHLEN], match[55], *peer;

    if (sc == NULL)
	return 1;
    if ((peer = peer_ip_as_string(net)) == NULL)
	return 2;
    dir = opendir(TLS_SFC_DIR);
    /* if the TLS_SFC_DIR dir doesn't exist, we consider this function disabled */
    if (dir == NULL)
	return 0;
    snprintf(match, sizeof(match), "oss%s", peer);
    match_len = strlen(match);
    /* search the dir for files matching the peer's ip address */
    while ((de = readdir(dir))) {
	if (!strncmp(de->d_name, match, match_len)) {
	    snprintf(filename, sizeof(filename), "%s/%s", TLS_SFC_DIR, de->d_name);
	    file = fopen(filename, "r");
	    if (file) {
		SSL_SESSION *sess;
		sess = PEM_read_SSL_SESSION(file, NULL, NULL, NULL);
		if (sess) {
		    /* ``refresh'' the session timeout */
/*XXX		    SSL_SESSION_set_time(sess, time(NULL)); */
		    if (session_timed_out(sess))
			/* a safer way to delete the old file perhaps? */
			unlink(filename);
		    else
			SSL_CTX_add_session(sc, sess);
		    /* dec the ref counter in sess so it will eventually be freed */
		    SSL_SESSION_free(sess);
		}
		fclose(file);
	    }
	}
    }
    closedir(dir);
    return 0;
}

int tls_sfc_new_session_cb(SSL * ssl, SSL_SESSION * sess)
{
    FILE *file;
    file = make_sfc_file();
    if (file) {
        PEM_write_SSL_SESSION(file, sess);
        fclose(file);
    }
    return 0;
}
#endif /* TLS_SESSION_FILE_CACHE */

int tls_init(void)
{
    int err;

#ifdef DEBUG_OPENSSL_MEM
    CRYPTO_malloc_debug_init();
    CRYPTO_mem_ctrl(CRYPTO_MEM_CHECK_ON);
#endif /* DEBUG_OPENSSL_MEM */
    SSL_load_error_strings();
    SSL_library_init();
    if (seed_PRNG())
	SYSLOG(LOG_ERR, "Wasn't able to properly seed the PRNG!");
        /* XXX Shouldn't we bail out here??? */
#ifdef ZLIB
    {
	COMP_METHOD *cm = COMP_zlib();
	if (!tls_nozlib && cm != NULL && cm->type != NID_undef) {
	    SSL_COMP_add_compression_method(0xe0, cm); /* Eric Young's ZLIB ID */
	}
    }
#endif /* ZLIB */
    ssl_ctx = SSL_CTX_new(SSLv23_server_method());
    if (!ssl_ctx) {
	SYSLOG(LOG_ERR, "SSL_CTX_new() %s",
		(char *)ERR_error_string(ERR_get_error(), NULL));
	return 1;
    }
#ifdef PR_TELNETD
    /* XXX SSL_OP_ALL breaks zlib compression with K95, why? */
    SSL_CTX_set_options(ssl_ctx, SSL_OP_NO_SSLv2);
#else
    SSL_CTX_set_options(ssl_ctx, SSL_OP_ALL | SSL_OP_NO_SSLv2);
#endif /* !PR_TELNETD */
    if (tls_client_cert_req)
	SSL_CTX_set_verify(ssl_ctx, SSL_VERIFY_PEER | SSL_VERIFY_FAIL_IF_NO_PEER_CERT,
			   verify_callback);
    else if (!tls_dont_request_cert)
	SSL_CTX_set_verify(ssl_ctx, SSL_VERIFY_PEER, verify_callback);
    SSL_CTX_set_default_verify_paths(ssl_ctx);
#if defined(PR_OBSD_FTPD) || defined(PR_PROFTPD)
    /* set up session caching  */
    SSL_CTX_set_session_cache_mode(ssl_ctx, SSL_SESS_CACHE_SERVER);
    SSL_CTX_set_session_id_context(ssl_ctx, (const unsigned char *) "1", 1);
#endif /* PR_OBSD_FTPD || PR_PROFTPD */

    /* let's find out which files are available */
#ifdef PR_PROFTPD
    tls_set_defaults();
    PRIVS_ROOT	/* ProFTPD must regain root privs here to read the key file(s) */
#endif /* PR_PROFTPD */
    check_file(&tls_rsa_cert_file);
    check_file(&tls_rsa_cert_chain_file);
    check_file(&tls_rsa_key_file);
    check_file(&tls_dsa_cert_file);
    check_file(&tls_dsa_cert_chain_file);
    check_file(&tls_dsa_key_file);
    check_file(&tls_crl_file);
    check_file(&tls_dhparam_file);
    if (!tls_rsa_cert_file && !tls_rsa_cert_chain_file &&
		!tls_dsa_cert_file && !tls_dsa_cert_chain_file) {
    	SYSLOG(LOG_ERR, "No certificate files found!");
	return 2;
    }
    
    if (tls_rsa_cert_file) {
	err = SSL_CTX_use_certificate_file(ssl_ctx, tls_rsa_cert_file, X509_FILETYPE_PEM);
	if (err <= 0) {
	    SYSLOG(LOG_ERR, "SSL_CTX_use_certificate_file(%s) %s", tls_rsa_cert_file,
		(char *)ERR_error_string(ERR_get_error(), NULL));
	    return 3;
	}
	if (!tls_rsa_key_file)
	    tls_rsa_key_file = tls_rsa_cert_file;
    }
    /* if you are using a chain file, the server's cert is supposed to be included
     * first in the file, and takes presence over a cert file.
     */
    if (tls_rsa_cert_chain_file) {
	err = SSL_CTX_use_certificate_chain_file(ssl_ctx, tls_rsa_cert_chain_file);
	if (err <= 0) {
	    SYSLOG(LOG_ERR, "SSL_CTX_use_certificate_chain_file(%s) %s\n",
		    tls_rsa_cert_chain_file,
		    (char *)ERR_error_string(ERR_get_error(), NULL));
	    return 4;
	}
	if (!tls_rsa_key_file || tls_rsa_key_file == tls_rsa_cert_file)
	    tls_rsa_key_file = tls_rsa_cert_chain_file;
    }
    if (tls_rsa_key_file) {
	err = SSL_CTX_use_PrivateKey_file(ssl_ctx, tls_rsa_key_file, X509_FILETYPE_PEM);
	if (err <= 0) {
	    SYSLOG(LOG_ERR, "SSL_CTX_use_PrivateKey_file(%s) %s", tls_rsa_key_file,
	    	(char *)ERR_error_string(ERR_get_error(), NULL));
	    return 5;
	}
    }
    if (tls_dsa_cert_file) {
	err = SSL_CTX_use_certificate_file(ssl_ctx, tls_dsa_cert_file, X509_FILETYPE_PEM);
	if (err <= 0) {
	    SYSLOG(LOG_ERR, "SSL_CTX_use_certificate_file(%s) %s", tls_dsa_cert_file,
		(char *)ERR_error_string(ERR_get_error(), NULL));
	    return 6;
	}
	if (!tls_dsa_key_file)
	    tls_dsa_key_file = tls_dsa_cert_file;
    }
    if (tls_dsa_cert_chain_file) {
	err = SSL_CTX_use_certificate_chain_file(ssl_ctx, tls_dsa_cert_chain_file);
	if (err <= 0) {
	    SYSLOG(LOG_ERR, "SSL_CTX_use_certificate_chain_file(%s) %s\n",
		    tls_dsa_cert_chain_file,
		    (char *)ERR_error_string(ERR_get_error(), NULL));
	    return 7;
	}
	if (!tls_dsa_key_file || tls_dsa_key_file == tls_dsa_cert_file)
	    tls_dsa_key_file = tls_dsa_cert_chain_file;
    }
    if (tls_dsa_key_file) {
	err = SSL_CTX_use_PrivateKey_file(ssl_ctx, tls_dsa_key_file, X509_FILETYPE_PEM);
	if (err <= 0) {
	    SYSLOG(LOG_ERR, "SSL_CTX_use_PrivateKey_file(%s) %s", tls_dsa_key_file,
	    	(char *)ERR_error_string(ERR_get_error(), NULL));
	    return 8;
	}
    }
#ifdef PR_PROFTPD
    PRIVS_RELINQUISH	/* ProFTPD dropping root privs again */
#endif /* PR_PROFTPD */

    SSL_CTX_set_tmp_rsa_callback(ssl_ctx, tmp_rsa_cb);
    SSL_CTX_set_tmp_dh_callback(ssl_ctx, tmp_dh_cb);

    /* set up the CRL */
    if ((tls_crl_file || tls_crl_dir) && (crl_store = X509_STORE_new()))
	X509_STORE_load_locations(crl_store, tls_crl_file, tls_crl_dir);

#ifdef TLS_SESSION_FILE_CACHE
    SSL_CTX_set_session_id_context(ssl_ctx, (const unsigned char *) "1", 1);
    tls_sfc_server_load(ssl_ctx);
    SSL_CTX_sess_set_new_cb(ssl_ctx, tls_sfc_new_session_cb);
#endif /* TLS_SESSION_FILE_CACHE */
    if (tls_cipher_list)
	SSL_CTX_set_cipher_list(ssl_ctx, tls_cipher_list);
    else
	SYSLOG(LOG_NOTICE, "NULL tls_cipher_list!");

#ifdef PR_TELNETD
    ssl = SSL_new(ssl_ctx);
    if (!ssl) {
	SYSLOG(LOG_ERR, "SSL_new() %s\n",
		(char *)ERR_error_string(ERR_get_error(), NULL));
	return 9;
    }
    SSL_set_fd(ssl, net);
#endif /* PR_TELNETD */

    return 0;
}

char *tls_userid_from_client_cert(void)
{
    static char cn[256];
    static char *r = cn;
    static int again = 0;
    int err;
    X509 *client_cert;

#if defined(PR_PROFTPD)
    if (!tls_on_ctrl || !first_ssl)
#elif defined(PR_OBSD_FTPD)
    if (!ctrl_conn.ssl)
#elif defined(PR_TELNETD)
    if (!tls_active)
#endif
    	return NULL;
    if (again)
    	return r;
    again = 1;
#if defined(PR_PROFTPD)
    if ((client_cert = SSL_get_peer_certificate(first_ssl))) {
#elif defined(PR_OBSD_FTPD)
    if ((client_cert = SSL_get_peer_certificate(ctrl_conn.ssl))) {
#elif defined(PR_TELNETD)
    if ((client_cert = SSL_get_peer_certificate(ssl))) {
#endif
    	/* call the custom function */
	err = x509_to_user(client_cert, cn, sizeof(cn));
	X509_free(client_cert);
	if (err)
	    return r = NULL;
	else
	    return r;
    } else
	return r = NULL;
}

int tls_is_user_valid(char *user)
/* check if clients cert is in "user"'s ~/.tlslogin file */
{
    char buf[512];
    int r = 0;
    FILE *fp = NULL;
    X509 *client_cert = NULL, *file_cert;
    struct passwd *pwd;

#if defined(PR_PROFTPD)
    if (!tls_on_ctrl || !first_ssl || !user)
#elif defined(PR_OBSD_FTPD)
    if (!ctrl_conn.ssl || !user)
#elif defined(PR_TELNETD)
    if (!user)
#endif
	return 0;

#ifdef PR_PROFTPD
    PRIVS_ROOT	/* ProFTPD must regain root privs here to read the .tlslogin file */
#endif /* PR_PROFTPD */
    if (!(pwd = getpwnam(user)))
    	goto cleanup;
    snprintf(buf, sizeof(buf), "%s/.tlslogin", pwd->pw_dir);
    if (!(fp = fopen(buf, "r")))
    	goto cleanup;
#if defined(PR_PROFTPD)
    if (!(client_cert = SSL_get_peer_certificate(first_ssl)))
#elif defined(PR_OBSD_FTPD)
    if (!(client_cert = SSL_get_peer_certificate(ctrl_conn.ssl)))
#elif defined(PR_TELNETD)
    if (!(client_cert = SSL_get_peer_certificate(ssl)))
#endif
    	goto cleanup;
    while ((file_cert = PEM_read_X509(fp, NULL, NULL, NULL))) {
	if (!M_ASN1_BIT_STRING_cmp(client_cert->signature, file_cert->signature))
	    r = 1;
	X509_free(file_cert);
	if (r) break;
    }
  cleanup:
    if (client_cert)
	X509_free(client_cert);
    if (fp)
	fclose(fp);
#ifdef PR_PROFTPD
    PRIVS_RELINQUISH	/* ProFTPD dropping root privs again */
#endif /* PR_PROFTPD */
    return r;
}

#ifdef PR_OBSD_FTPD
void tls_close_session(CONN *conn)
{
    if (!conn->ssl)
	return;
    SSL_shutdown(conn->ssl);
    SSL_free(conn->ssl);
    conn->ssl = NULL;
    conn->sock = -1;
}

int tls_accept(int s, int dataconn)
{
    int err;
    static int logged_data_connection = 0;
    CONN *conn;

    if (dataconn)
	conn = &data_conn;
    else
	conn = &ctrl_conn;
    if (conn->ssl) {
	SYSLOG(LOG_ERR, "Already TLS connected!");
	return 1;
    }
    conn->ssl = SSL_new(ssl_ctx);
    if (!conn->ssl) {
	SYSLOG(LOG_ERR, "SSL_new() %s", (char *)ERR_error_string(ERR_get_error(), NULL));
	return 2;
    }
    SSL_set_fd(conn->ssl, s);
    conn->sock = s;
    
  retry:
    err = SSL_accept(conn->ssl);
    if (err < 1) {
	int ssl_err = SSL_get_error(conn->ssl, err);
	SYSLOG(LOG_ERR, "SSL_accept(): (%d) %s", ssl_err,
		(char *) ERR_error_string(ERR_get_error(), NULL));
	if (ssl_err == SSL_ERROR_WANT_READ || ssl_err == SSL_ERROR_WANT_WRITE)
	    goto retry;
	tls_close_session(conn);
	return 3;
    }
   
    /* only log first TLS data connection, otherwise there might be lots of logging */
    if (dataconn) {
	if (!logged_data_connection) {
	    SYSLOG(LOG_NOTICE, "TLS data connection using cipher %s (%d bits)",
		   SSL_get_cipher(conn->ssl), SSL_get_cipher_bits(conn->ssl, NULL));
	    logged_data_connection = 1;
	}
    } else {
	char *subject = tls_get_subject_name(conn->ssl);
	SYSLOG(LOG_NOTICE, "TLS connection using cipher %s (%d bits)",
	       SSL_get_cipher(conn->ssl), SSL_get_cipher_bits(conn->ssl, NULL));
	if (subject)
	    SYSLOG(LOG_NOTICE, "Client: %s", subject);
    }
    return 0;
}

void tls_shutdown(void)
{
    if (data_conn.ssl) {
    	SSL_shutdown(data_conn.ssl);
	SSL_free(data_conn.ssl);
	data_conn.ssl = NULL;
	data_conn.sock = -1;
    }
    if (ctrl_conn.ssl) {
    	SSL_shutdown(ctrl_conn.ssl);
	SSL_free(ctrl_conn.ssl);
	ctrl_conn.ssl = NULL;
	ctrl_conn.sock = -1;
    }
}
#endif /* PR_OBSD_FTPD */

#ifdef PR_PROFTPD
void tls_close_session(SSL *ssl)
{
    if (!ssl)
	return;
    SSL_shutdown(ssl);
    SSL_free(ssl);
}

int tls_accept(conn_t *conn, int data_connection)
{
    int err;
    SSL *ssl;
    static int logged_data_connection = 0;

    if (!ssl_ctx) {
	SYSLOG(LOG_ERR, "tls_accept() called when ssl_ctx == NULL!");
	return 1;
    }
    ssl = SSL_new(ssl_ctx);
    if (!ssl) {
	SYSLOG(LOG_ERR, "SSL_new() %s", (char *)ERR_error_string(ERR_get_error(), NULL));
	return 2;
    }

    if (!first_ssl)
	first_ssl = ssl;
    SSL_set_fd(ssl, conn->rfd); /* it works with either rfd or wfd (I hope ;-) */
    
  retry:
    err = SSL_accept(ssl);
    if (err < 1) {
	int ssl_err = SSL_get_error(ssl, err);
	if (ssl_err == SSL_ERROR_WANT_READ || ssl_err == SSL_ERROR_WANT_WRITE)
	    goto retry;
	SYSLOG(LOG_ERR, "SSL_accept(): (%d) %s", ssl_err,
		(char *)ERR_error_string(ERR_get_error(), NULL));
	tls_close_session(ssl);
	return 3;
    }
    conn->inf->tls_session = conn->outf->tls_session = ssl;
    tls_on_ctrl = 1;
    
    if (data_connection) {
	/* only log first TLS data connection, otherwise there might be lots of logging */
	if (!logged_data_connection) {
	    SYSLOG(LOG_NOTICE, "TLS data connection using cipher %s (%d bits)",
		    SSL_get_cipher(ssl), SSL_get_cipher_bits(ssl, NULL));
	    logged_data_connection = 1;
	}
    } else {
	char *subject = tls_get_subject_name(first_ssl);
	SYSLOG(LOG_NOTICE, "TLS connection using cipher %s (%d bits)",
		SSL_get_cipher(ssl), SSL_get_cipher_bits(ssl, NULL));
	if (subject)
	    SYSLOG(LOG_NOTICE, "Client: %s", subject);
    }
    return 0;
}
#endif /* PR_PROFTPD */

#ifdef PR_TELNETD
void tls_shutdown(void)
{
    if (tls_active) {
    	SSL_shutdown(ssl);
    	tls_active = 0;
    }
}

int tls_accept(void)
{
    int err;
    char *subject;
	
    err = SSL_accept(ssl);
    if (err < 1) {
	SYSLOG(LOG_INFO, "SSL_accept() %s\n", (char *)ERR_error_string(ERR_get_error(), NULL));
	tls_shutdown();
	return 1;
    }
/*    SYSLOG(LOG_INFO, "SSL_get_verify_result() = %d\n", SSL_get_verify_result(ssl));*/
    tls_active = 1;
    SYSLOG(LOG_INFO, "TLS connection using cipher %s (%d bits)", SSL_get_cipher(ssl),
    	SSL_get_cipher_bits(ssl, NULL));
    subject = tls_get_subject_name(ssl);
    if (subject)
	SYSLOG(LOG_NOTICE, "Client: %s", subject);
    return 0;
}
#endif /* PR_TELNETD */

void tls_cleanup(void)
{
#ifdef PR_OBSD_FTPD
    tls_shutdown();
#endif /* PR_OBSD_FTPD */
    if (crl_store) {
    	X509_STORE_free(crl_store);
	crl_store = NULL;
    }
#ifdef PR_TELNETD
    if (ssl) {
	SSL_free(ssl);
	ssl = NULL;
    }
#endif /* PR_TELNETD */
    if (ssl_ctx) {
	SSL_CTX_free(ssl_ctx);
	ssl_ctx = NULL;
    }
    if (tmp_dh) {
	DH_free(tmp_dh);
	tmp_dh = NULL;
    }
    if (tmp_rsa) {
	RSA_free(tmp_rsa);
	tmp_rsa = NULL;
    }
    ERR_free_strings();
    ERR_remove_state(0);
    EVP_cleanup();	/* release the stuff allocated by SSL_library_init() */
    if (tls_rsa_key_file) {
    	if (tls_rsa_key_file != tls_rsa_cert_file)
	    free(tls_rsa_key_file);
	tls_rsa_key_file = NULL;
    }
    if (tls_rsa_cert_file) {
    	free(tls_rsa_cert_file);
	tls_rsa_cert_file = NULL;
    }
    if (tls_rsa_cert_chain_file) {
    	free(tls_rsa_cert_chain_file);
	tls_rsa_cert_chain_file = NULL;
    }
    if (tls_dsa_key_file) {
    	if (tls_dsa_key_file != tls_dsa_cert_file)
	    free(tls_dsa_key_file);
	tls_dsa_key_file = NULL;
    }
    if (tls_dsa_cert_file) {
    	free(tls_dsa_cert_file);
	tls_dsa_cert_file = NULL;
    }
    if (tls_dsa_cert_chain_file) {
    	free(tls_dsa_cert_chain_file);
	tls_dsa_cert_chain_file = NULL;
    }
    if (tls_dhparam_file) {
    	free(tls_dhparam_file);
	tls_dhparam_file = NULL;
    }
    if (tls_crl_file) {
    	free(tls_crl_file);
	tls_crl_file = NULL;
    }
    if (tls_crl_dir) {
    	free(tls_crl_dir);
	tls_crl_dir = NULL;
    }
    if (tls_cipher_list) {
    	free(tls_cipher_list);
	tls_cipher_list = NULL;
    }
    if (tls_rand_file)
	/* tls_rand_file is not malloc()'ed */
	RAND_write_file(tls_rand_file);
#ifdef DEBUG_OPENSSL_MEM
    {
#if defined(PR_OBSD_FTPD)
    char fname[] = "/tmp/ftpd_memleak_XXXXXX";
#elif defined(PR_PROFTPD)
    char fname[] = "/tmp/proftpd_memleak_XXXXXX";
#elif defined(PR_TELNETD)
    char fname[] = "/tmp/telnetd_memleak_XXXXXX";
#endif
    int fd;
    if ((fd = mkstemp(fname)) != -1) {
    	FILE *f = fdopen(fd, "w");
        if (f) {
	    CRYPTO_mem_leaks_fp(f);
	    fclose(f);
	}
    }
    }
#endif /* DEBUG_OPENSSL_MEM */
}

static void handle_ssl_error(int error, char *where)
{
    switch (error) {
    	case SSL_ERROR_NONE:
	    return;
	case SSL_ERROR_SSL:
	    SYSLOG(LOG_ERR, "Panic: SSL_ERROR_SSL in %s!", where);
	    break;
	case SSL_ERROR_WANT_READ:
	    SYSLOG(LOG_ERR, "Panic: SSL_ERROR_WANT_READ in %s!", where);
	    break;
	case SSL_ERROR_WANT_WRITE:
	    SYSLOG(LOG_ERR, "Panic: SSL_ERROR_WANT_WRITE in %s!", where);
	    break;
	case SSL_ERROR_WANT_X509_LOOKUP:
	    SYSLOG(LOG_ERR, "Panic: SSL_ERROR_WANT_X509_LOOKUP in %s!", where);
	    break;
	case SSL_ERROR_SYSCALL:
#ifdef PR_PROFTPD
	    /* ProFTPD is sloppy in checking the return value/errno status from
	     * read()/write(), so we must double-check here so we don't generate
	     * unnecessary fuzz...
	     */
	    if (errno == ECONNRESET) /* peer hung up */
		return;
#endif /* PR_PROFTPD */
	    SYSLOG(LOG_ERR, "Panic: SSL_ERROR_SYSCALL in %s!", where);
	    break;
	case SSL_ERROR_ZERO_RETURN:
	    SYSLOG(LOG_ERR, "Panic: SSL_ERROR_ZERO_RETURN in %s!", where);
	    break;
	case SSL_ERROR_WANT_CONNECT:
	    SYSLOG(LOG_ERR, "Panic: SSL_ERROR_WANT_CONNECT in %s!", where);
	    break;
	default:
	    SYSLOG(LOG_ERR, "Panic: SSL_ERROR %d in %s!", error, where);
	    break;
    }
    /* if we reply something here, we might just trigger another handle_ssl_error()
     * call and loop endlessly...
     */
#if defined(PR_PROFTPD)
    main_exit((void*) LOG_ERR,
	      "Unexpected OpenSSL error, disconnected.", (void*) 0, NULL);
    /* NOTREACHED */
#elif defined (PR_OBSD_FTPD)
    syslog(LOG_ERR, "Unexpected OpenSSL error, disconnected.");
    dologout(error);
    /* NOTREACHED */
#endif
}

static int select_read(int rfd)
/* timeout = 20 seconds */
{
    fd_set rfds;
    struct timeval tv;

    FD_ZERO(&rfds);
    FD_SET(rfd, &rfds);
    tv.tv_sec = 20;
    tv.tv_usec = 0;
    return select(rfd + 1, &rfds, NULL, NULL, &tv);
}

#if defined(PR_OBSD_FTPD) || defined(PR_TELNETD)
ssize_t tls_read(int fd, void *buf, size_t count)
{
# ifdef PR_OBSD_FTPD
    SSL *ssl = SOCK_TO_SSL(fd);
# endif /* PR_OBSD_FTPD */
#elif defined(PR_PROFTPD)
ssize_t tls_read(SSL *ssl, void *buf, size_t count)
{
#endif

  retry:
#if defined(PR_OBSD_FTPD) || defined(PR_PROFTPD)
    if (ssl) {
#elif defined(PR_TELNETD)
    if (tls_active) {
#endif
	ssize_t c = SSL_read(ssl, buf, count);
#ifdef PR_TELNETD
	DIAG(TD_TLSDATA, { if (c > 0) printdata("sd<", buf, c); });
#endif /* PR_TELNETD */
	if (c < 0) {
	    int err = SSL_get_error(ssl, c);
	    /* read(2) returns only the generic error number -1 */
	    c = -1;
	    switch (err) {
		case SSL_ERROR_WANT_READ:
		    /* OpenSSL needs more data from the wire to finish the current block,
		     * so we wait a little while for it. */
#if defined(PR_OBSD_FTPD) || defined(PR_TELNETD)
		    err = select_read(fd);
#elif defined(PR_PROFTPD)
		    err = select_read(SSL_get_fd(ssl));
#endif
		    if (err > 0)
			goto retry;
		    else if (err == 0)
			/* still missing data after timeout. simulate an EINTR and return. */
			errno = EINTR;
		    /* if err < 0, i.e. some error from the select(), everything is already
		     * in place; errno is properly set and this function returns -1. */
		    break;
		case SSL_ERROR_SSL:
		case SSL_ERROR_SYSCALL:
		    /* the SSL connection is screwed up, emulate EOF */
		    c = 0;
		    break;
		default:
		    handle_ssl_error(err, "tls_read()");
		    break;
	    }
	}
	return c;
    } else
#if defined(PR_OBSD_FTPD) || defined(PR_TELNETD)
	return read(fd, buf, count);
#elif defined(PR_PROFTPD)
	return 0;	/* this should never happen */
#endif
}
 
#if defined(PR_OBSD_FTPD) || defined(PR_TELNETD)
ssize_t tls_write(int fd, const void *buf, size_t count)
{
# ifdef PR_OBSD_FTPD
    SSL *ssl = SOCK_TO_SSL(fd);
# endif /* PR_OBSD_FTPD */
#elif defined(PR_PROFTPD)
ssize_t tls_write(SSL *ssl, const void *buf, size_t count)
{
#endif

#if defined(PR_OBSD_FTPD) || defined(PR_PROFTPD)
    if (ssl) {
#elif defined(PR_TELNETD)
    if (tls_active) {
#endif
	ssize_t c = SSL_write(ssl, buf, count);
#ifdef PR_TELNETD
	DIAG(TD_TLSDATA, { if (c > 0 && diagnostic & TD_TO_FILE) printdata("sd>", buf, c); });
#endif /* PR_TELNETD */
	if (c < 0) {
	    int err = SSL_get_error(ssl, c);
	    /* write(2) returns only the generic error number -1 */
	    c = -1;
	    switch (err) {
		case SSL_ERROR_WANT_WRITE:
		    /* simulate an EINTR in case OpenSSL wants to write more */
		    errno = EINTR;
		    break;
		case SSL_ERROR_SSL:
		case SSL_ERROR_SYSCALL:
		    /* the SSL connection is screwed up, emulate EPIPE */
		    errno = EPIPE;
		    break;
		default:
		    handle_ssl_error(err, "tls_write()");
		    break;
	    }
	}
	return c;
    } else
#if defined(PR_OBSD_FTPD) || defined(PR_TELNETD)
	return write(fd, buf, count);
#elif defined(PR_PROFTPD)
	return 0;	/* this should never happen */
#endif
}

#ifdef PR_PROFTPD 
int tls_pending(SSL *ssl)
{
    if (ssl)
        return SSL_pending(ssl);
    else 
        return 0;
}
#endif /* PR_PROFTPD */

#ifdef PR_TELNETD
int tls_recv(int s, void *buf, size_t len, int flags)
{
    if (tls_active)
	return (int)tls_read(s, buf, len);
    else
	return recv(s, buf, len, flags);
}

int tls_send(int s, const void *msg, size_t len, int flags)
{
    if (tls_active)
	return (int)tls_write(s, msg, len);
    else
	return send(s, msg, len, flags);
}

inline int tls_pending(void)
{
    if (tls_active)
        return SSL_pending(ssl);
    else 
        return 0;
}
#endif /* PR_TELNETD */
 
#ifdef PR_OBSD_FTPD
#ifdef __STDC__
int tls_fprintf(FILE *stream, const char *fmt, ...)
#else
int tls_fprintf(stream, fmt, va_alist)
    FILE *stream;
    char *fmt;
    va_dcl
#endif
{
    SSL *ssl = SOCK_TO_SSL(fileno(stream));
    va_list ap;
    
#ifdef __STDC__
    va_start(ap, fmt);
#else
    va_start(ap);
#endif
    if (ssl) {
	/* we _should_ do this with a dynamic buffer, but that relies on a proper
	 * vsnprintf() implementation, ie one that returns the number of bytes it
	 * _wants_ to write, regardless of the output buffer size. this is not
	 * true for glibc < 2.1, so we use a static buffer for now. but we should
	 * fix it for OpenBSD.
	 */
	char buf[1024];
	int sent = 0, size, w;
	vsnprintf(buf, sizeof(buf), fmt, ap);
	size = strlen(buf);
	do {
	    w = tls_write(fileno(stream), buf + sent, size - sent);
	    if (w > 0)
		sent += w;
	    else if (!(w < 0 && errno == EINTR))
		break;	/* other error than EINTR or w == 0 */
        } while (sent != size);
	return sent;
    } else
	return vfprintf(stream, fmt, ap);
}

int tls_vfprintf(FILE *stream, const char *format, va_list ap)
{
#define SNP_MAXBUF 1024000
    SSL *ssl = SOCK_TO_SSL(fileno(stream));

    if (ssl) {
	/* here I boldly assume that snprintf() and vsnprintf() uses the same
	 * return value convention. if not, what kind of libc is this? ;-)
	 */
	char sbuf[1024] = { 0 }, *buf = sbuf, *lbuf = NULL;
	int sent = 0, size, ret, w;
	ret = vsnprintf(sbuf, sizeof(sbuf), format, ap);
#ifdef SNPRINTF_OK
	/* this one returns the number of bytes it wants to write in case of overflow */
	if (ret >= sizeof(sbuf) && ret < SNP_MAXBUF) {
	    /* sbuf was too small, use a larger lbuf */
	    lbuf = malloc(ret + 1);
	    if (lbuf) {
		vsnprintf(lbuf, ret + 1, format, ap);
		buf = lbuf;
	    }
	}
#else
# ifdef SNPRINTF_HALFBROKEN
	/* this one returns the number of bytes written (excl. \0) in case of overflow */
#  define SNP_OVERFLOW(x, y) ( x == y ? 1 : 0 )
#  define SNP_NOERROR(x)     ( x < 0 ? 0 : 1 )
# else
#  ifdef SNPRINTF_BROKEN
	/* this one returns -1 in case of overflow */
#   define SNP_OVERFLOW(x, y) ( x < 0 ? 1 : 0 )
#   define SNP_NOERROR(x)     ( 1 )  /* if -1 means overflow, what's the error indication? */
#  else
#   error No valid SNPRINTF_... macro defined!
#  endif /* !SNPRINTF_BROKEN */
# endif /* !SNPRINTF_HALFBROKEN */
	if (SNP_NOERROR(ret) && SNP_OVERFLOW(ret, sizeof(sbuf) - 1)) {
	    /* sbuf was too small, use a larger lbuf */
	    size = sizeof(sbuf);
	    do {
		if ((size *= 2) > SNP_MAXBUF)	/* try to double the size */
		    break;
		if (lbuf) free(lbuf);
		lbuf = malloc(size);
		if (lbuf) {
		    ret = vsnprintf(lbuf, size, format, ap);
		    buf = lbuf;
		} else
		    break;
	    } while (SNP_NOERROR(ret) && SNP_OVERFLOW(ret, size - 1));
	}
#endif /* !SNPRINTF_OK */
	size = strlen(buf);
	do {
	    w = tls_write(fileno(stream), buf + sent, size - sent);
	    if (w > 0)
		sent += w;
	    else if (!(w < 0 && errno == EINTR))
		break;	/* other error than EINTR or w == 0 */
        } while (sent != size);
	if (lbuf) free(lbuf);
	return sent;
    } else
	return vfprintf(stream, format, ap);
}

#ifdef __STDC__
int tls_printf(const char *fmt, ...)
#else
int tls_printf(fmt, va_alist)
    char *fmt;
    va_dcl
#endif
{
    va_list ap;
#ifdef __STDC__
    va_start(ap, fmt);
#else
    va_start(ap);
#endif
    return tls_vfprintf(stdout, fmt, ap);
}

int tls_vprintf(const char *format, va_list ap)
{
    return tls_vfprintf(stdout, format, ap);
}

int tls_fgetc(FILE *stream)
{
    SSL *ssl = SOCK_TO_SSL(fileno(stream));

    if (ssl) {
	unsigned char r;
	int err;
	do
	    err = tls_read(fileno(stream), &r, 1);
	while (err < 0 && errno == EINTR);
	if (err == 1)
	    return (int) r;
	else
	    return EOF;
    } else
	return fgetc(stream);
}

int tls_fputc(int c, FILE *stream)
{
    SSL *ssl = SOCK_TO_SSL(fileno(stream));

    if (ssl) {
	unsigned char uc = c;
	int err;
	do
	    err = tls_write(fileno(stream), &uc, 1);
	while (err < 0 && errno == EINTR);
	if (err == 1)
	    return (int) uc;
	else
	    return EOF;
    } else
	return fputc(c, stream);
}

int tls_fclose(FILE *stream)
{
    SSL *ssl = SOCK_TO_SSL(fileno(stream));
    
    if (ssl) {
    	SSL_shutdown(ssl);
	SSL_free(ssl);
	if (ssl == data_conn.ssl) {
	    data_conn.ssl = NULL;
	    data_conn.sock = -1;
	} else if (ssl == ctrl_conn.ssl) {
	    ctrl_conn.ssl = NULL;
	    ctrl_conn.sock = -1;
	}
    }
    return fclose(stream);
}

int tls_close(int fd)
{
    SSL *ssl = SOCK_TO_SSL(fd);
    
    if (ssl) {
    	SSL_shutdown(ssl);
	SSL_free(ssl);
	if (ssl == data_conn.ssl) {
	    data_conn.ssl = NULL;
	    data_conn.sock = -1;
	} else if (ssl == ctrl_conn.ssl) {
	    ctrl_conn.ssl = NULL;
	    ctrl_conn.sock = -1;
	}
    }
    return close(fd);
}

int tls_fflush(FILE *stream)
{
    if (stream == NULL)
	return fflush(NULL);
    if (SOCK_TO_SSL(fileno(stream)))
	return 0;	/* don't do anything! */
    else
	return fflush(stream);
}

void tls_exit(int status)
{
    tls_cleanup();
    exit(status);
}
#endif /* PR_OBSD_FTPD */
